/*
libhugepagealloc - wrapper malloc for using large pages 
Copyright (C) 2005  Simon Hausmann

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

Contact: jan.treibig@informatik.uni-erlangen.de
*/

#include <assert.h>
#include <sys/param.h>
#include <string.h>
#include <errno.h>
#include <stdio.h>
#include <sys/mman.h>
#include <unistd.h>
#include <fcntl.h>
#include <stdlib.h>
#include <stdio.h>
#include <paths.h>
#include <mntent.h>

#define debug(string) \
    write(2, string, strlen(string));

/* #### */
#if !defined(__USE_GNU)
#define __USE_GNU
#endif
#include <dlfcn.h>

// ### all not thread safe

static void *(*oldMalloc)(size_t size);
static void *(*oldFree)(void *ptr);
static int (*oldPosixMemAlign)(void **memptr, size_t alignment, size_t size);

static void init()
{
    if (!oldMalloc)
        oldMalloc = (void *(*)(size_t))dlsym(RTLD_NEXT, "malloc");

    if (!oldFree)
        oldFree = (void *(*)(void *))dlsym(RTLD_NEXT, "free");

    if (!oldPosixMemAlign)
        oldPosixMemAlign = (int (*)(void **, size_t, size_t))dlsym(RTLD_NEXT, "posix_memalign");
}

typedef struct HugePageInfo HugePageInfo;
struct HugePageInfo
{
    void *start;
    size_t length;
};

#define REALLY_LAME_MAX_ALLOC 1000
static HugePageInfo allocations[REALLY_LAME_MAX_ALLOC];

static HugePageInfo *findFreeSlot()
{
    int i;
    for (i = 0; i < REALLY_LAME_MAX_ALLOC; ++i)
        if (!allocations[i].start)
            return &allocations[i];

    write(2, "UGH\n", 4);

    return 0;
}

static HugePageInfo *findSlotForAddr(void *ptr)
{
    int i;
    for (i = 0; i < REALLY_LAME_MAX_ALLOC; ++i)
        if (allocations[i].start == ptr)
            return &allocations[i];

    return 0;
}

static const char *tlbfsMountPoint()
{
    static char mnt[PATH_MAX+1];
    static int checked = 0;

    if (checked == 0) {
        FILE *mnttab = setmntent(_PATH_MOUNTED, "r");
        struct mntent *entry;
        mnt[0] = 0;

        checked = 1;

        if (!mnttab) {
            fprintf(stderr, "setmntent on %s failed!\n", _PATH_MOUNTED);
            exit(1);
        }

        entry = getmntent(mnttab);
        while (entry) {
            if (!strcmp(entry->mnt_type, "hugetlbfs")) {
                assert(strlen(entry->mnt_dir) < PATH_MAX);

                strcpy(mnt, entry->mnt_dir);
                break;
            }

            entry = getmntent(mnttab);
        }

        endmntent(mnttab);

        if (mnt[0] == 0) {
            fprintf(stderr, "cannot find any hugetlbfs filesystem!\n");
            exit(1);
        }
    }
    return mnt;
}

static void *hugePageAlloc(size_t size, HugePageInfo *handle)
{
    static int inAllocation = 0;
    static int infoPrinted = 0;
    const int page = 1 << 22;
    int fd;
    const char suffix[] = "/XXXXXX";
    char file[PATH_MAX+1];

    if (inAllocation != 0) {
        debug("hugePageAlloc is not reentrant!\n");
        exit(1);
    }

    inAllocation = 1;

    strcpy(file, tlbfsMountPoint());
    assert(strlen(file) < PATH_MAX - strlen(suffix));
    strcat(file, suffix);

    fd = mkstemp(file);
    if (fd == -1) {
        perror("hugePageAlloc(mkstemp)");
        exit(1);
    }
    unlink(file);

    size = ((size + page) / page ) * page;

    handle->start = mmap(0UL, size, PROT_READ | PROT_WRITE, MAP_SHARED, fd, 0);
    handle->length = size;
    if (handle->start == (void *)MAP_FAILED) {
        perror("hugePageAlloc(mmap)");
        exit(1);
    }
    close(fd);
    if (infoPrinted == 0) {
        debug("large page allocation active\n");
        infoPrinted = 1;
    }
    inAllocation = 0;
    return handle->start;
}

static void freeHugePage(HugePageInfo *handle)
{
    if (munmap(handle->start, handle->length) == -1)
        perror("freeHugePage(munmap)");
}

void *malloc(size_t size);
void *malloc(size_t size)
{
    init();

    const size_t page = 1 << 22;
    if (size < page) {
        return oldMalloc(size);
    }

    return hugePageAlloc(size, findFreeSlot());
};

void *calloc(size_t nmemb, size_t size);
void *calloc(size_t nmemb, size_t size)
{
    size *= nmemb;
    void *ptr = 0;

    init();

    const size_t page = 1 << 22;
    if (size < page) {
        ptr = oldMalloc(size);
    } else {
        ptr = hugePageAlloc(size, findFreeSlot());
    }
    if (ptr)
        memset(ptr, 0, size);
    return ptr;
};

void free(void *ptr);
void free(void *ptr)
{
    HugePageInfo *handle;

    init();

    handle = findSlotForAddr(ptr);
    if (!handle) {
        oldFree(ptr);
        return;
    }
    freeHugePage(handle);
    handle->start = 0;
};

int posix_memalign(void **memptr, size_t alignment, size_t size);
int posix_memalign(void **memptr, size_t alignment, size_t size)
{
    init();

    const size_t page = 1 << 22;
    if (size < page || alignment > page) {
        return oldPosixMemAlign(memptr, alignment, size);
    }

    *memptr = hugePageAlloc(size, findFreeSlot());
    return 0;
};

/* vim: et sw=4 ts=4
 */
