#include "arrayInfo.h"
#include <assert.h>
#include <limits.h>

// CW: constructor to generate an array info:
// elemSize: size of an array element
// address: base address of array
// columnSizes: size of all columns (first vector element is size of dimension 1 and so on)
// bound: lower bound (base) of the array for each dimension (first vector element is lower bound of dimension 1 and so on)
arrayInfo::arrayInfo(unsigned long elemSize,unsigned long address,const std::vector<int>& columnSizes, const std::vector<int>& bounds): baseAddress(address),elementSize(elemSize),dimensions(columnSizes.size()),colSize(columnSizes),lowerBound(bounds)
{
  assert(colSize.size()==dimensions);
  assert(lowerBound.size()==dimensions);
}

arrayInfo::arrayInfo(const arrayInfo& rhs):
  baseAddress(rhs.baseAddress),elementSize(rhs.elementSize),dimensions(rhs.dimensions), colSize(rhs.colSize),lowerBound(rhs.lowerBound),references(rhs.references)
{}

arrayInfo& arrayInfo::operator=(const arrayInfo& rhs)
{
  if(this==&rhs) return *this;
	
  baseAddress=rhs.baseAddress;
  elementSize=rhs.elementSize;
  dimensions=rhs.dimensions;
  colSize=rhs.colSize;
  lowerBound=rhs.lowerBound;
  references=rhs.references;
  
  return *this;
}

// CW: nothing to do so far
arrayInfo::~arrayInfo()
{}

// CW: return column size: dim is dimension starting from 1
int arrayInfo::getLowerBound(unsigned long dim) const
{
  assert(dim <= dimensions);
  return lowerBound[dim-1];
}


// CW: set column size: dim is dimension starting from 1
void arrayInfo::setColSize(unsigned long dim, int size)
{
  assert(dim <= dimensions);
  assert(size>0);
	
  colSize[dim-1]=size;
}

// CW: returns size of array in bytes
unsigned long arrayInfo::getSize() const
{
  unsigned long result=colSize[0];
  for(unsigned long i=1; i<getNumberOfDimensions(); i++)
    {
      assert(result< ULONG_MAX/colSize[i]);
      
      result *= colSize[i];
    }
	
  assert(result < ULONG_MAX/getElementSize());
  return result * getElementSize();
}

// CW: add information of a new reference to array
void arrayInfo::addReference(const arrayReference& newRef)
{
  references.insert(references.end(),newRef);
}

// CW: add information of a new reference to array
void arrayInfo::addReferences(const std::vector<arrayReference>& newRefs)
{
  references.insert(references.end(),newRefs.begin(),newRefs.end());
}

bool arrayInfo::operator==(const arrayInfo& rhs) const
{
  return (this==&rhs)
    ||( (baseAddress==rhs.baseAddress)
	&& (elementSize==rhs.elementSize)
	&& (dimensions==rhs.dimensions)
	&& (colSize==rhs.colSize)
	&& (lowerBound==rhs.lowerBound)
	&& (references==rhs.references));
}

bool arrayInfo::operator!=(const arrayInfo& rhs) const
{
  return !((*this)==(rhs));
}
