#include <math.h>
#include <assert.h>
#include "dpUtilities.h"
#include <sys/types.h>
#include <sys/times.h>
 
// Creation of a filename for debugging purposes
// it creates a filename of the form "identifier.lvl<level>.iter<iteration>.dat"
// if the parameter iteration is equal to zero (default value) the
// filename is "identifier.lvl<level>.dat" instead
const std::string dpGetFileName(const char *identifier,int level,int iteration)
{
  std::string fName(identifier);
  fName+=".lvl";
  fName+=('0'+level); // hack will only work for numbers smaller than 10

  if(iteration!=0){
    fName+=".iter";
    fName+=('0'+iteration);
  };

  fName+=".dat";

  return fName;
}


void dpPrintGrid(ostream& os,dpGrid2d& grid)
{
  dpPrintGrid(os,grid.getmem(),grid.getdimx(),grid.getdimy(),grid.getpad());
}

void dpPrintGrid(ostream& os,DIME_REAL *mem,int nxp,int nyp,int pad)
{
  for(int x=0; x<nxp ; x++)
  {
    for(int y=0; y<nyp ; y++)
    {
      os << x << " " << y << " " << mem[y*(nxp+pad)+x] << endl;
    }

    os << endl;
  }
}


// The following function computes a suitable relaxation parameter for
// red-black SOR smoothers when applied to certain elliptic operators,
// see I. Yavneh, On Red-Black SOR Smoothing in Multigrid,
// SIAM J. SCI. COMPUT., Vol. 17, No. 1, pp. 180-192, January 1996,
// We merely use this technique in the case of anisotropic Laplace problems

DIME_REAL dpCalcOmega(DIME_REAL hx, DIME_REAL hy)
{
  DIME_REAL xisq, ooxisq, cxTilde, cyTilde, cMin, CMax, omega;

  assert(hx>0.0);
  assert(hy>0.0);

  xisq= (hx*hx)/(hy*hy);
  ooxisq= 1.0/xisq;
  cxTilde= ooxisq/(1.0+ooxisq);
  cyTilde= 1.0-cxTilde;
  (cxTilde<cyTilde) ? cMin= cxTilde : cMin= cyTilde;
  CMax= 1.0-cMin;
  omega= 2.0/(1.0+sqrt(1.0-CMax*CMax));

#ifndef DIME_NDEBUG
  cout << "DiMEPACK: Omega: " << omega << endl;
  cout << "DiMEPACK: cxTilde: " << cxTilde << endl;
  cout << "DiMEPACK: cyTilde: " << cyTilde << endl;
#endif

  return omega;
}


// This function determines the maximum number of grid levels
// including the level whose dimensions are nxp and nyp:
int dpNumLevelsLimit(int nxp, int nyp)
{
  if (nxp>=3 && nyp>=3 && nxp%2==1 && nyp%2==1)
    return 1 + dpNumLevelsLimit((nxp-1)/2+1,(nyp-1)/2+1);
  else
    return 0;
}

void dpPrintTimes(const char *id)
{
#ifdef DIME_TIMING_ENABLED
  struct tms buffer;

  times(&buffer);
  float userTimeElapsed=(float)buffer.tms_utime/(float)CLK_TCK;
  float systemTimeElapsed=(float)buffer.tms_stime/(float)CLK_TCK;
 
  cout << "DiMEPACK TIMING";
  cout << " (" << id << "): ";
  cout <<  userTimeElapsed << "u " << systemTimeElapsed << "s " << endl;
#endif
}
