#include <assert.h>
#include <stdlib.h>
#include "dpRestrictRhs.h"
#include "dpUtilities.h"
#include <fstream.h>

// forward declaration of utility functions
static void dpRestrictRhsLevelFW(int nxpFine, int nypFine, int padFine, DIME_REAL *fFine,
				 int nxpCoarse, int nypCoarse, int padCoarse, DIME_REAL *fCoarse);
static void dpRestrictRhsLevelHW(int nxpFine, int nypFine, int padFine, DIME_REAL *fFine,
				 int nxpCoarse, int nypCoarse, int padCoarse, DIME_REAL *fCoarse);
static void dpRestrictRhsBoundaryFW(int nxpFine, int nypFine, int padFine, DIME_REAL *fFine,
				  int nxpCoarse, int nypCoarse, int padCoarse, DIME_REAL *fCoarse);
static void dpRestrictRhsBoundaryHW(int nxpFine, int nypFine, int padFine, DIME_REAL *fFine,
				  int nxpCoarse, int nypCoarse, int padCoarse, DIME_REAL *fCoarse);

inline DIME_REAL dpCalcRhsNodeFW(DIME_REAL ce,DIME_REAL no,DIME_REAL ne,DIME_REAL ea,DIME_REAL se,DIME_REAL so,DIME_REAL sw,DIME_REAL we,DIME_REAL nw)
{
  const DIME_REAL centerCoeff=1.0;
  const DIME_REAL diagCoeff=0.25;
  const DIME_REAL horizCoeff=0.5;
  
  return ce*centerCoeff+horizCoeff*(no+ea+so+we)+diagCoeff*(ne+se+sw+nw);
}

inline DIME_REAL dpCalcRhsNodeHW(DIME_REAL ce,DIME_REAL no,DIME_REAL ea,DIME_REAL so,DIME_REAL we)
{
  const DIME_REAL centerCoeff=2.0;
  const DIME_REAL horizCoeff=0.5;

  return ce*centerCoeff+horizCoeff*(no+ea+so+we);
}

void dpRestrictRhs(const dpMgData& dat, tRestrict restType)
{
  for(int i=dat.maxLevel;i>0;i--)
  {
    assert(dat.fGrids[i]!=NULL);
    assert(dat.fGrids[i-1]!=NULL);

    
#ifdef DIME_DEBUG_FMGRHS
    ofstream outfile(dpGetFileName("f-fmg",i).c_str());
    dpPrintGrid(outfile,dat.fGrids[i],dat.xDims[i],dat.yDims[i],dat.fPads[i]);
#endif
    
    switch(restType){
    case FW: // full weighting
      dpRestrictRhsLevelFW(dat.xDims[i],dat.yDims[i],dat.fPads[i],dat.fGrids[i],
			   dat.xDims[i-1],dat.yDims[i-1],dat.fPads[i-1],dat.fGrids[i-1]);
      break;

    case HW: // half weighting
      dpRestrictRhsLevelHW(dat.xDims[i],dat.yDims[i],dat.fPads[i],dat.fGrids[i],
			   dat.xDims[i-1],dat.yDims[i-1],dat.fPads[i-1],dat.fGrids[i-1]);
      break;

    default:
      exit(-1);
      break;
    };
  }

#ifdef DIME_DEBUG_FMGRHS
    ofstream outfile(dpGetFileName("f-fmg",0).c_str());
    dpPrintGrid(outfile,dat.fGrids[0],dat.xDims[0],dat.yDims[0],dat.fPads[0]);
#endif

}

void dpRestrictRhsLevelFW(int nxpFine, int nypFine, int padFine, DIME_REAL *fFine, int nxpCoarse, int nypCoarse, int padCoarse, DIME_REAL *fCoarse)
{
  // perform restriction on inner part of rhs

  for(int y=1;y<nypCoarse-1;y++){
    for(int x=1;x<nxpCoarse-1;x++){

      int xFine=2*x;
      int yFine=2*y;
      
      fCoarse[y*(nxpCoarse+padCoarse)+x]=dpCalcRhsNodeFW(
	fFine[ yFine    *(nxpFine+padFine) + xFine     ],  // CE
	fFine[ (yFine+1)*(nxpFine+padFine) + xFine     ],  // NO
	fFine[ (yFine+1)*(nxpFine+padFine) + (xFine+1) ],  // NE
	fFine[ yFine    *(nxpFine+padFine) + (xFine+1) ],  // EA
	fFine[ (yFine-1)*(nxpFine+padFine) + (xFine+1) ],  // SE
	fFine[ (yFine-1)*(nxpFine+padFine) + xFine     ],  // SO
	fFine[ (yFine-1)*(nxpFine+padFine) + (xFine-1) ],  // SW
	fFine[ yFine    *(nxpFine+padFine) + (xFine-1) ],  // WE
	fFine[ (yFine+1)*(nxpFine+padFine) + (xFine-1) ]); // NW
    }
  }

  dpRestrictRhsBoundaryFW(nxpFine,nypFine,padFine,fFine,nxpCoarse,nypCoarse,padCoarse,fCoarse);
}


void dpRestrictRhsLevelHW(int nxpFine, int nypFine, int padFine, DIME_REAL *fFine, int nxpCoarse, int nypCoarse, int padCoarse, DIME_REAL *fCoarse)
{
  // perform restrictin on inner part of rhs
  for(int y=1;y<nypCoarse-1;y++){
    for(int x=1;x<nxpCoarse-1;x++){
      
      int xFine=2*x;
      int yFine=2*y;
      
      fCoarse[y*(nxpCoarse+padCoarse)+x]=dpCalcRhsNodeHW(
        fFine[ yFine    *(nxpFine+padFine) + xFine     ],  // CE
	fFine[ (yFine+1)*(nxpFine+padFine) + xFine     ],  // NO
	fFine[ yFine    *(nxpFine+padFine) + (xFine+1) ],  // EA
	fFine[ (yFine-1)*(nxpFine+padFine) + xFine     ],  // SO
	fFine[ yFine    *(nxpFine+padFine) + (xFine-1) ]); // WE
    }
  }
  
  dpRestrictRhsBoundaryHW(nxpFine,nypFine,padFine,fFine,nxpCoarse,nypCoarse,padCoarse,fCoarse);
}
 
// perform restriction on boundaries, this is done via half injection: 0.5 * 4.0 = 2.0,
// this is not necessary when we have Dirichlet boundaries
void dpRestrictRhsBoundaryFW(int nxpFine, int nypFine, int padFine, DIME_REAL *fFine, int nxpCoarse, int nypCoarse, int padCoarse, DIME_REAL *fCoarse)
{
  const DIME_REAL ce=1.0;
  const DIME_REAL di=0.5;

  for(int x=1;x<nxpCoarse-1;x++){
    int xFine=2*x;

    // south boundary
    fCoarse[x]=
      ce*fFine[xFine]+ // ce
      ce*fFine[(nxpFine+padFine)+xFine]+ // no
      di*fFine[xFine-1]+ // we
      di*fFine[xFine+1]+ // ea
      di*fFine[(nxpFine+padFine)+xFine-1]+ // nw
      di*fFine[(nxpFine+padFine)+xFine+1]; // we

    // north boundary
    fCoarse[(nypCoarse-1)*(nxpCoarse+padCoarse)+x]=
      ce*fFine[(nypFine-1)*(nxpFine+padFine)+xFine]+ // ce
      ce*fFine[(nypFine-2)*(nxpFine+padFine)+xFine]+ // so
      di*fFine[(nypFine-1)*(nxpFine+padFine)+xFine-1]+ // we
      di*fFine[(nypFine-1)*(nxpFine+padFine)+xFine+1]+ // ea
      di*fFine[(nypFine-2)*(nxpFine+padFine)+xFine-1]+ // sw
      di*fFine[(nypFine-2)*(nxpFine+padFine)+xFine+1]; // se
  }

  for(int y=1;y<nypCoarse-1;y++){
    int yFine=2*y;

    // west boundary
    fCoarse[y*(nxpCoarse+padCoarse)]=
      ce*fFine[yFine*(nxpFine+padFine)]+ // ce
      ce*fFine[yFine*(nxpFine+padFine)+1]+ // ea
      di*fFine[(yFine-1)*(nxpFine+padFine)+1]+ // se
      di*fFine[(yFine+1)*(nxpFine+padFine)+1]+ // ne
      di*fFine[(yFine-1)*(nxpFine+padFine)]+ // so
      di*fFine[(yFine+1)*(nxpFine+padFine)]; // no

    // east boundary
    fCoarse[y*(nxpCoarse+padCoarse)+nxpCoarse-1]=
      ce*fFine[yFine*(nxpFine+padFine)+nxpFine-1]+ // ce
      ce*fFine[yFine*(nxpFine+padFine)+nxpFine-2]+ // we
      di*fFine[(yFine-1)*(nxpFine+padFine)+nxpFine-2]+ // sw
      di*fFine[(yFine+1)*(nxpFine+padFine)+nxpFine-2]+ // nw
      di*fFine[(yFine-1)*(nxpFine+padFine)+nxpFine-1]+ // so
      di*fFine[(yFine+1)*(nxpFine+padFine)+nxpFine-1]; // no
  }

  // sw corner
  fCoarse[0]=
    ce*fFine[0]+ // ce
    ce*fFine[1]+ // ea
    ce*fFine[nxpFine+padFine]+ // no
    ce*fFine[nxpFine+padFine+1]; // ne

  // se corner
  fCoarse[nxpCoarse-1]=
    ce*fFine[nxpFine-1]+ // ce
    ce*fFine[nxpFine-2]+ // we
    ce*fFine[2*nxpFine+padFine-1]+ // no
    ce*fFine[2*nxpFine+padFine-2]; // nw

  // nw corner
  fCoarse[(nypCoarse-1)*(nxpCoarse+padCoarse)]=
    ce*fFine[(nypFine-1)*(nxpFine+padFine)]+ // ce
    ce*fFine[(nypFine-1)*(nxpFine+padFine)+1]+ // ea
    ce*fFine[(nypFine-2)*(nxpFine+padFine)]+ // so
    ce*fFine[(nypFine-2)*(nxpFine+padFine)+1]; // se

  // ne corner
  fCoarse[(nypCoarse-1)*(nxpCoarse+padCoarse)+nxpCoarse-1]=
    ce*fFine[(nypFine-1)*(nxpFine+padFine)+nxpFine-1]+ // ce
    ce*fFine[(nypFine-1)*(nxpFine+padFine)+nxpFine-2]+ // we
    ce*fFine[(nypFine-2)*(nxpFine+padFine)+nxpFine-1]+ // so
    ce*fFine[(nypFine-2)*(nxpFine+padFine)+nxpFine-2]; // sw
}

void dpRestrictRhsBoundaryHW(int nxpFine, int nypFine, int padFine, DIME_REAL *fFine, int nxpCoarse, int nypCoarse, int padCoarse, DIME_REAL *fCoarse)
{
  const DIME_REAL ce=2.0;
  const DIME_REAL ho=1.0;
  const DIME_REAL ve=0.5;

  for(int x=1;x<nxpCoarse-1;x++){
    int xFine=2*x;

    // south boundary
    fCoarse[x]=
      ce*fFine[xFine]+ // ce
      ho*fFine[(nxpFine+padFine)+xFine]+ // no
      ve*fFine[xFine-1]+ // we
      ve*fFine[xFine+1]; // ea

    // north boundary
    fCoarse[(nypCoarse-1)*(nxpCoarse+padCoarse)+x]=
      ce*fFine[(nypFine-1)*(nxpFine+padFine)+xFine]+ // ce
      ho*fFine[(nypFine-2)*(nxpFine+padFine)+xFine]+ // so 
      ve*fFine[(nypFine-1)*(nxpFine+padFine)+xFine-1]+ // we
      ve*fFine[(nypFine-1)*(nxpFine+padFine)+xFine+1]; // ea
  }

  for(int y=1;y<nypCoarse-1;y++){
    int yFine=2*y;

    // west boundary
    fCoarse[y*(nxpCoarse+padCoarse)]=
      ce*fFine[yFine*(nxpFine+padFine)]+ // ce
      ho*fFine[yFine*(nxpFine+padFine)+1]+ // ea
      ve*fFine[(yFine-1)*(nxpFine+padFine)]+ // so
      ve*fFine[(yFine+1)*(nxpFine+padFine)]; // no

    // east boundary
    fCoarse[y*(nxpCoarse+padCoarse)+nxpCoarse-1]=
      ce*fFine[yFine*(nxpFine+padFine)+nxpFine-1]+ // ce
      ho*fFine[yFine*(nxpFine+padFine)+nxpFine-2]+ // we
      ve*fFine[(yFine-1)*(nxpFine+padFine)+nxpFine-1]+ // so
      ve*fFine[(yFine+1)*(nxpFine+padFine)+nxpFine-1]; // no
  }

  // sw corner
  fCoarse[0]=
    ce*fFine[0]+ // ce
    ho*fFine[1]+ // ea
    ho*fFine[nxpFine+padFine]; // no

  // se corner
  fCoarse[nxpCoarse-1]=
    ce*fFine[nxpFine-1]+ // ce
    ho*fFine[nxpFine-2]+ // we 
    ho*fFine[2*nxpFine+padFine-1]; // no

  // nw corner
  fCoarse[(nypCoarse-1)*(nxpCoarse+padCoarse)]=
    ce*fFine[(nypFine-1)*(nxpFine+padFine)]+ // ce
    ho*fFine[(nypFine-1)*(nxpFine+padFine)+1]+ // ea
    ho*fFine[(nypFine-2)*(nxpFine+padFine)]; // so

  // ne corner
  fCoarse[(nypCoarse-1)*(nxpCoarse+padCoarse)+nxpCoarse-1]=
    ce*fFine[(nypFine-1)*(nxpFine+padFine)+nxpFine-1]+ // ce
    ho*fFine[(nypFine-1)*(nxpFine+padFine)+nxpFine-2]+ // we
    ho*fFine[(nypFine-2)*(nxpFine+padFine)+nxpFine-1]; // so
  
}
