#include <stdio.h>
#include <fstream.h>
#include <assert.h>
#include <math.h>
#include <stdlib.h>
#include <string.h>
#include "dpMgData.h"
#include "dpApplyPadding.h"
#include "dpDirectSolve.h"
#include "dpUtilities.h"

#define min(a,b) (a<b) ? a : b

dpMgData::dpMgData(int nLevels, int nCoeff, dpGrid2d *u, dpGrid2d *fIn, DIME_REAL *matCoeff,
		   tBoundary *bTypes, bool isInitialized, DIME_REAL **bVals, DIME_REAL omega,
		   bool isFmg, const bool fixCGSolution):
  coeff(new DIME_REAL[nCoeff]),
  fLoc(NULL)
{
  assert(coeff!=NULL);
  assert(u!=NULL);
  assert(matCoeff!=NULL);
  assert(nCoeff==5 || nCoeff==9);
  this->nCoeff= nCoeff;
  this->matCoeff= matCoeff;

  // Check if size of the finest grid is valid:
  assert(u->getdimx()>4);
  assert(u->getdimy()>4);
  if ((u->getdimx()-1)%4 != 0 || (u->getdimy()-1)%4 != 0) {
    cout << "DiMEPACK: Finest grid has invalid size!" << endl;
    exit (-1);
  }

  // Use maximum number of grid levels, if necessary:
  int nLevelsLimit= dpNumLevelsLimit(u->getdimx(),u->getdimy());

#ifdef DIME_DEBUG_NUMLEVELS
  cout << "DiMEPACK: Maximum number of levels (nLevelsLimit): " << nLevelsLimit << endl;
#endif 
  	
  if  (nLevels==1) {
    // This is not allowed since we do not permit padding on the coarsest level
    // of the hierarchy:
    cerr << "DiMEPACK: One level is not allowed! Using " << nLevelsLimit << " levels" << endl;
    nLevels= nLevelsLimit;
  }
  else {
    if (nLevels==0) {
      // User requires the maximum number of grid levels:
      cout << "DiMEPACK: Number of levels unspecified! Using " << nLevelsLimit << " levels" << endl;
      nLevels= nLevelsLimit;
    }
    else {
      // User wants to choose the number of levels himself/herself:
      if (nLevels>nLevelsLimit) {
	// User has specified a number of levels which is too large:
	cout << "DiMEPACK: Too many levels required! Using " << nLevelsLimit << " levels" << endl;
	nLevels= nLevelsLimit;
      }
    }
  }

#ifdef DIME_DEBUG_NUMLEVELS
  cout << "DiMEPACK: Actual number of levels (nLevels): " << nLevels << endl;
#endif 

  // Define variables:
  xDims= new int[nLevels];
  yDims= new int[nLevels];
  uPads= new int[nLevels];
  fPads= new int[nLevels];
  hx= new DIME_REAL[nLevels];
  hy= new DIME_REAL[nLevels];
  uGrids= new DIME_REAL*[nLevels];
  fGrids= new DIME_REAL*[nLevels];
  maxLevel= nLevels-1;

  // Determine dimensions of finest level:
  xDims[maxLevel]= u->getdimx();
  yDims[maxLevel]= u->getdimy();
  uPads[maxLevel]= u->getpad();
  hx[maxLevel]= u->gethx();
  hy[maxLevel]= u->gethy();

  assert(hx[maxLevel]>0.0);
  assert(hy[maxLevel]>0.0);

  // Determine if problem is homogeneous:
  if (fIn==NULL && bTypes[dpNORTH]==DIRICHLET && bTypes[dpEAST]==DIRICHLET &&
      bTypes[dpSOUTH]==DIRICHLET && bTypes[dpWEST]==DIRICHLET)
    isHom= true;
  else
    {
      // We need a copy of the rhs:
      fLoc= new dpGrid2d(xDims[maxLevel], yDims[maxLevel],
			 hx[maxLevel], hy[maxLevel]);
      assert(fLoc!=NULL);
      fGrids[maxLevel]= fLoc->getmem();
      fPads[maxLevel]= fLoc->getpad();
      isHom= false;
      if (fIn==NULL)
	     fLoc->initzero();
      else
	     (*fLoc)=(*fIn);
    }

  // Check if grid size is ok:
  for(int l= maxLevel-1; l>=0; l--)
    {
      assert((xDims[l+1]%2)==1);
      xDims[l]= (xDims[l+1]-1)/2+1;
      assert((yDims[l+1]%2)==1);
      yDims[l]= (yDims[l+1]-1)/2+1;
      assert(xDims[l]>=3);
      assert(yDims[l]>=3);
      hx[l]= hx[l+1]*2.0;
      hy[l]= hy[l+1]*2.0;
    }

  // Determine suitable padding constants:
  dpApplyPadding(maxLevel, xDims, yDims, uPads, fPads); 

  // Padding on coarsest level is forbidden!
  uPads[0]= 0;
  fPads[0]= 0;
  
  // Access to the finest level, initialize solution vector (if necessary):
  uGrids[maxLevel]= u->getmem();
  if(isInitialized==false)
    u->initzero();
  
  // Create coarser grids, array padding must be introduced according to
  // the paddings defined a few lines above:
  for(int l= maxLevel-1; l>=0; l--)
    {
      uGrids[l]= new DIME_REAL[(xDims[l]+uPads[l])*yDims[l]];
      // Solution vector must be cleared in the FMG context so that interpolation
      //gets a correct start value:
      if(isFmg) memset(uGrids[l],0,(xDims[l]+uPads[l])*yDims[l]*sizeof(DIME_REAL));
      fGrids[l]= new DIME_REAL[(xDims[l]+fPads[l])*yDims[l]];

#ifndef NDEBUG
      for (int c= 0; c<(xDims[l]+fPads[l])*yDims[l]; c++)
			fGrids[l][c]= 1000000;
#endif
    }
  
  // Set up the matrix for the coarsest system:
  if (nCoeff==5) {
#ifndef DIME_NDEBUG 
    cout << "DiMEPACK: 5-point stencil specified" << endl;
#endif
    dpBuild5pCMatrix(xDims[0], yDims[0], matCoeff, omega, bTypes, fixCGSolution);
  }
  else {
#ifndef DIME_NDEBUG 
    cout << "DiMEPACK: 9-point stencil specified" << endl;
#endif
    dpBuild9pCMatrix(xDims[0], yDims[0], matCoeff, omega, bTypes, fixCGSolution);
  }
  
  // Determine if the non-center coefficients are equal:
  if (nCoeff==5)	
    { // matCoeff[2] is the diagonal entry
      if(matCoeff[0]==matCoeff[1] && matCoeff[1]==matCoeff[3] &&
	 matCoeff[3]==matCoeff[4])
	isIdCoeff= true;
      else
	isIdCoeff= false;
    }
  else
    { // matCoeff[4] is the diagonal entry
      if((matCoeff[0]==matCoeff[2] && matCoeff[2]==matCoeff[6] &&
	  matCoeff[6]==matCoeff[8]) &&
	 (matCoeff[1]==matCoeff[3] && matCoeff[3]==matCoeff[5] &&
	  matCoeff[5]==matCoeff[7]))
	isIdCoeff= true;
      else
	isIdCoeff= false;
    }

  // Compute the coefficients for smoothing and residual calculation procedures,
  // prepare the rhs, and respect Dirichlet boundary conditions, if necessary:
  getCoeffConst(omega);
  if (isHom==false)
    prepRhsConst(fGrids[maxLevel], xDims[maxLevel], yDims[maxLevel],
		 fPads[maxLevel], omega, bTypes,
		 bVals, hx[maxLevel], hy[maxLevel]);
  insertDBCConst(uGrids[maxLevel], xDims[maxLevel], yDims[maxLevel],
		 uPads[maxLevel], bTypes, bVals);
  
#ifndef DIME_NDEBUG
  {
 	 ofstream outfile("u-init.dat");
 	 dpPrintGrid(outfile, u->getmem(), u->getdimx(), u->getdimy(), u->getpad());
  }
#endif
}  


dpMgData::~dpMgData() {
  delete [] xDims;
  delete [] yDims;
  delete [] uPads;
  delete [] fPads;
  delete [] hx;
  delete [] hy;
  delete [] coeff;

  dpDirectSolveCleanup();

  for (int l=maxLevel-1; l>=0; l--) {
    delete [] uGrids[l];
    delete [] fGrids[l];
  }
  delete [] uGrids;
  delete [] fGrids;

  if (fLoc)
    delete fLoc;
}


// Method getCoeffConst
// --------------------
// Computes the coefficients that are passed e.g. to the smoothing
// procedures
//
// Parameters:
// -----------
// omega:    Relaxation parameter

void dpMgData::getCoeffConst(DIME_REAL omega)
{
  if (nCoeff==5)
    {
      // matCoeff[2] is the diagonal entry
      coeff[0]= 1.0-omega;
      coeff[1]= -omega*matCoeff[4]/matCoeff[2];
      coeff[2]= -omega*matCoeff[3]/matCoeff[2];
      coeff[3]= -omega*matCoeff[0]/matCoeff[2];
      coeff[4]= -omega*matCoeff[1]/matCoeff[2];
    }
  else
    { // matCoeff[4] is the diagonal entry
      coeff[0]= 1.0-omega;
      coeff[1]= -omega*matCoeff[7]/matCoeff[4];
      coeff[2]= -omega*matCoeff[8]/matCoeff[4];
      coeff[3]= -omega*matCoeff[5]/matCoeff[4];
      coeff[4]= -omega*matCoeff[2]/matCoeff[4];
      coeff[5]= -omega*matCoeff[1]/matCoeff[4];
      coeff[6]= -omega*matCoeff[0]/matCoeff[4];
      coeff[7]= -omega*matCoeff[3]/matCoeff[4];
      coeff[8]= -omega*matCoeff[6]/matCoeff[4];
    }
  
  return;
}


// Method prepRhsConst
// -------------------
// Preprocesses the rhs e.g. for the smoothing procedures
//
// Parameters:
// -----------
// f:        Rhs
// nxp, nyp: Numbers of grid points in dimensions x and y, respectively
// fpad:     Padding in array f
// nCoeff:   Number of entries per row of the matrix
// matCoeff: Array of coefficients (matrix entries)
// omega:    Relaxation parameter
// bTypes:   Array specifying the types of the boundary conditions
// bVals:    Boundary values
// hx, hy:   Mesh widths in dimensions x and y, respectively

void dpMgData::prepRhsConst(DIME_REAL *f, int nxp, int nyp, int fpad,
			    DIME_REAL omega, tBoundary *bTypes,
			    DIME_REAL **bVals, DIME_REAL hx, DIME_REAL hy)
{
  int i, x, y, center= ((nCoeff==5) ? 2 : 4);
  DIME_REAL h;

  if (nCoeff==5)
    {
      // Non-corner boundary nodes:
      // North boundary:
      if (bTypes[dpNORTH]==NEUMANN)
	for(x= 1; x<nxp-1; x++)
	  f[(nyp-1)*(nxp+fpad)+x]-= 2*hy*bVals[dpNORTH][x]*matCoeff[4];
      
      // South boundary:
      if (bTypes[dpSOUTH]==NEUMANN)
	for(x= 1; x<nxp-1; x++)
	  f[x]-= 2*hy*bVals[dpSOUTH][x]*matCoeff[0];
      
      // East boundary:
      if (bTypes[dpEAST]==NEUMANN)
	for(y= 1; y<nyp-1; y++)
	  f[y*(nxp+fpad)+nxp-1]-= 2*hx*bVals[dpEAST][y]*matCoeff[3];
      
      // West boundary:
      if (bTypes[dpWEST]==NEUMANN)
	for(y= 1; y<nyp-1; y++)
	  f[y*(nxp+fpad)]-= 2*hx*bVals[dpWEST][y]*matCoeff[1];
      
      // Corner nodes:
      // North-west corner:
      if (bTypes[dpNORTH]==NEUMANN && bTypes[dpWEST]==NEUMANN)
	f[(nyp-1)*(nxp+fpad)]-= 2*(matCoeff[4]*hy*bVals[dpNORTH][0]+
				   matCoeff[1]*hx*bVals[dpWEST][nyp-1]);

      // North-east corner:
      if (bTypes[dpNORTH]==NEUMANN && bTypes[dpEAST]==NEUMANN)
	f[(nyp-1)*(nxp+fpad)+nxp-1]-= 2*(matCoeff[4]*hy*bVals[dpNORTH][nxp-1]+
					 matCoeff[3]*hx*bVals[dpEAST][nyp-1]);

      // South-east corner:
      if (bTypes[dpEAST]==NEUMANN && bTypes[dpSOUTH]==NEUMANN)
	f[nxp-1]-= 2*(matCoeff[0]*hy*bVals[dpSOUTH][nxp-1]+
		      matCoeff[3]*hx*bVals[dpEAST][0]);

      // South-west corner:
      if (bTypes[dpSOUTH]==NEUMANN && bTypes[dpWEST]==NEUMANN)
	f[0]-= 2*(matCoeff[0]*hy*bVals[dpSOUTH][0]+
		  matCoeff[1]*hx*bVals[dpWEST][0]);
    }
  else
    {
      // Non-corner boundary nodes:
      // North boundary:
      if (bTypes[dpNORTH]==NEUMANN)
	for(x= 1; x<nxp-1; x++)
	  f[(nyp-1)*(nxp+fpad)+x]-= 2*hy*(bVals[dpNORTH][x-1]*matCoeff[6]+
					  bVals[dpNORTH][x]*matCoeff[7]+
					  bVals[dpNORTH][x+1]*matCoeff[8]);
      
      // South boundary:
      if (bTypes[dpSOUTH]==NEUMANN)
	for(x= 1; x<nxp-1; x++)
	  f[x]-= 2*hy*(bVals[dpSOUTH][x-1]*matCoeff[0]+
		       bVals[dpSOUTH][x]*matCoeff[1]+
		       bVals[dpSOUTH][x+1]*matCoeff[2]);
      
      // East boundary:
      if (bTypes[dpEAST]==NEUMANN)
	for(y= 1; y<nyp-1; y++)
	  f[y*(nxp+fpad)+nxp-1]-= 2*hx*(bVals[dpEAST][y-1]*matCoeff[2]+
					bVals[dpEAST][y]*matCoeff[5]+
					bVals[dpEAST][y+1]*matCoeff[8]);
      
       // West boundary:
      if (bTypes[dpWEST]==NEUMANN)
	for(y= 1; y<nyp-1; y++)
	  f[y*(nxp+fpad)]-= 2*hx*(bVals[dpWEST][y-1]*matCoeff[0]+
				  bVals[dpWEST][y]*matCoeff[3]+
				  bVals[dpWEST][y+1]*matCoeff[6]);
      
      // Corner nodes:
      // North-west corner:
      if (bTypes[dpNORTH]==NEUMANN && bTypes[dpWEST]==NEUMANN)
	f[(nyp-1)*(nxp+fpad)]-= 2*(matCoeff[7]*hy*bVals[dpNORTH][0]+
				   matCoeff[3]*hx*bVals[dpWEST][nyp-1]);

      // North-east corner:
      if (bTypes[dpNORTH]==NEUMANN && bTypes[dpEAST]==NEUMANN)
	f[(nyp-1)*(nxp+fpad)+nxp-1]-= 2*(matCoeff[7]*hy*bVals[dpNORTH][nxp-1]+
					 matCoeff[5]*hx*bVals[dpEAST][nyp-1]);

      // South-east corner:
      if (bTypes[dpEAST]==NEUMANN && bTypes[dpSOUTH]==NEUMANN)
	f[nxp-1]-= 2*(matCoeff[1]*hy*bVals[dpSOUTH][nxp-1]+
		      matCoeff[5]*hx*bVals[dpEAST][0]);
      
      // South-west corner:
      if (bTypes[dpSOUTH]==NEUMANN && bTypes[dpWEST]==NEUMANN)
	f[0]-= 2*(matCoeff[1]*hy*bVals[dpSOUTH][0]+
		  matCoeff[3]*hx*bVals[dpWEST][0]);
    }

  // Modify RHS w.r.t. relaxation parameter and diagonal entry:
  for (y= 0; y<nyp; y++)
    for (x= 0; x<nxp; x++)
      f[y*(nxp+fpad)+x]*= (omega/matCoeff[center]);
  
  return;
}


// Method insertDBCConst
// ---------------------
// Inserts Dirichlet boundary values into the solution vector
//
// Parameters:
// -----------
// u:        Solution array
// nxp, nyp: Numbers of grid points in dimensions x and y, respectively
// upad:     Padding in array u
// bTypes:   Array specifying the types of the boundary conditions
// bVals:    Boundary values

void dpMgData::insertDBCConst(DIME_REAL *u, int nxp, int nyp, int upad, tBoundary *bTypes,
		      DIME_REAL **bVals)
{
  int i, x, y;

  if (bTypes[dpNORTH]==DIRICHLET) // North boundary:
    for (x= 0; x<nxp; x++)
      u[(nyp-1)*(nxp+upad)+x]= bVals[dpNORTH][x];
  
  if (bTypes[dpSOUTH]==DIRICHLET) // South boundary:
    for(x= 0; x<nxp; x++)
      u[x]= bVals[dpSOUTH][x];
  
  if (bTypes[dpEAST]==DIRICHLET) // East boundary:
    for(y= 0; y<nyp; y++)
      u[y*(nxp+upad)+nxp-1]= bVals[dpEAST][y];
  
  if (bTypes[dpWEST]==DIRICHLET) // West boundary:
    for(y= 0; y<nyp; y++)
      u[y*(nxp+upad)]= bVals[dpWEST][y];

  return;
}
