include(fortran-interface.m4)

#include <iostream.h>
#include <stdlib.h>
#include <assert.h>
#include <math.h>
#include "dpDirectSolve.h"
#include "dpDirectSolveGlobals.h"
#include "dpLAPACK.h"


// Macros to compute the positions within the matrices which are passed
// to the LAPACK routines DGBTRF/SGBTRF (LU factorization of a general band
// matrix) and DPBTRF/SPBTRF (Cholesky factorization of a s.p.d. matrix,
// the upper triangle is stored):
#define dpGetIndexGB(a,b) ((b)*ldab)+2*kl+(a)-(b)
#define dpGetIndexPB(a,b) ((b)*ldab)+kl+(a)-(b)

// Choose the correct LAPACK routines depending on the data type that is used
// for floating-point numbers: double precision or single precision:
#ifdef DIME_DOUBLE
dpCreateDefine(XGBTRS,dgbtrs)
dpCreateDefine(XPBTRS,dpbtrs)
#else
dpCreateDefine(XGBTRS,sgbtrs)
dpCreateDefine(XPBTRS,spbtrs)
#endif


// Global variables declared in dpDirectSolveGlobals.h:
bool useCholesky; // Flag specifying if the matrices are s.p.d.
DIME_REAL *Ac; // Matrix for the coarsest system
int kl; // Number of subdiagonals (= number of superdiagonals)
int ldab; // Leading dimension of matrix array Ac
int neqs; // Number of equations in coarsest system
int *ipiv; // Pivot indices
DIME_REAL *w; // Rhs resp. solution vector for the coarsest system
int *eqmap; // To map the equation to the index of the array
bool fixCorner; // Flag indicating if we have to fix the value in the SW corner


// Function dpCheckProblem
// -----------------------
// Checks if the system matrix is irreducibly diagonally dominant and thus
// a regular M-matrix or not. If it isn't, a warning is printed to stderr.
// The matrix is assumed to be irreducible!
//
// Parameters:
// -----------
// nCoeff:   Number of coefficients 
// matCoeff: Matrix coefficients
// bTypes:   Array of boundary types

void dpCheckProblem(int nCoeff, DIME_REAL *matCoeff, tBoundary *bTypes)
{
  DIME_REAL offDiagSum= 0.0;
  DIME_REAL c= 0.0;

  // Exit if the user specifies the following non-standard case:
  // 1. There is at least one Dirichlet boundary, and
  // 2. The user wants to fix a solution on the corsest grid:
  if ((bTypes[dpNORTH]==DIRICHLET || bTypes[dpEAST]==DIRICHLET ||
       bTypes[dpSOUTH]==DIRICHLET || bTypes[dpWEST]==DIRICHLET) &&
      fixCorner==true) {
    cerr << "DiMEPACK: There is at least one Dirichlet boundary, and you want ";
    cerr << "to fix a solution on the coarsest grid! Exiting ..." << endl;
    exit(1);
  }

  // Compute the sum of the absolute values of the off-diagonal entries:
  if (nCoeff==5) {
    offDiagSum= fabs(matCoeff[0])+fabs(matCoeff[1])+fabs(matCoeff[3])+fabs(matCoeff[4]);
    c= fabs(matCoeff[2]); // Absolute value of the diagonal entry
  }
  else {
    offDiagSum= fabs(matCoeff[0])+fabs(matCoeff[1])+fabs(matCoeff[2])+fabs(matCoeff[3])+
      fabs(matCoeff[5])+fabs(matCoeff[6])+fabs(matCoeff[7])+fabs(matCoeff[8]);
    c= fabs(matCoeff[4]); // Absolute value of the diagonal entry
  }

  // Check if the matrix is irreducibly diagonally dominant:
  if(c>offDiagSum)
    // Matrix is idd:
    return;

  if(c==offDiagSum && (bTypes[dpNORTH]==DIRICHLET ||
		      bTypes[dpEAST]==DIRICHLET ||
		      bTypes[dpSOUTH]==DIRICHLET ||
		      bTypes[dpWEST]==DIRICHLET))
    // Matrix is idd:
    return;

  // The matrix might not be irreducibly diagonally dominant:
#ifdef DIME_DEBUG_DIRECTSOLVE
  cerr << "DiMEPACK: Matrix might not be irreducibly diagonally dominant!" << endl;
#endif

  // Let's hope that it works and go ahead in spite of these circumstances:
  return;
}


// Function dpGetNumEqs
// --------------------
// Determines the number of equations in the coarsest system
// (= the order of the coarsest system matrix)
//
// Parameters:
// -----------
// nxp, nyp: Numbers of grid points in the coarsest grid
// btypes:   Array of boundary types

int dpGetNumEqs(int nxp, int nyp, tBoundary *btypes)
{
  int ndbpts= 0; // Number of Dirichlet boundary points

  // Non-corner points:
  if (btypes[dpNORTH]==DIRICHLET)
    ndbpts+= nxp-2;
  if (btypes[dpEAST]==DIRICHLET)
    ndbpts+= nyp-2;
  if (btypes[dpSOUTH]==DIRICHLET)
    ndbpts+= nxp-2;
  if (btypes[dpWEST]==DIRICHLET)
    ndbpts+= nyp-2;

  // Corners:
  // North-east:
  if (btypes[dpNORTH]==DIRICHLET || btypes[dpEAST]==DIRICHLET)
    ++ndbpts;
  // South-east:
  if (btypes[dpEAST]==DIRICHLET || btypes[dpSOUTH]==DIRICHLET)
    ++ndbpts;
  // South-west:
  if (btypes[dpSOUTH]==DIRICHLET || btypes[dpWEST]==DIRICHLET)
    ++ndbpts;
  // North-west:
  if (btypes[dpNORTH]==DIRICHLET || btypes[dpWEST]==DIRICHLET)
    ++ndbpts;

  return(nxp*nyp-ndbpts);
}


// Funcion dpUseCholesky
// ---------------------
// The following function determines if Choleky's method can be applied
// for factorizing the matrix corresponding to the coarsest system
// of linear equations. Note that the matrix must be symmetric and
// positive definite so that Cholesky's method can be applied.
//
// Parameters:
// -----------
// ncoeff:   Number of coefficients (5 or 9)
// matcoeff: Array of coefficients (matrix entries)
// btypes:   Array of boundary types

void dpUseCholesky(int ncoeff, DIME_REAL *matcoeff, tBoundary *btypes)
{
  DIME_REAL ndiagsum= 0.0;
  int i, center= (ncoeff==5) ? 2 : 4;
  
  if (ncoeff==5)
    {	
      if((matcoeff[0]!=matcoeff[4]) || (matcoeff[1]!=matcoeff[3]))
	{ // Matrix is not symmetric:
	  useCholesky= false;
	  return;
	}
    }
  else
    {
      if((matcoeff[0]!=matcoeff[8]) || (matcoeff[1]!=matcoeff[7]) ||
	 (matcoeff[2]!=matcoeff[6]) || (matcoeff[3]!=matcoeff[5]))
	{ // Matrix is not symmetric:
	  useCholesky= false;
	  return;
	}
    }
  //  OK, matrix is at least symmetric. Let's go ahead ...

  // Check the diagonal entry:
  if (matcoeff[center]<0.0)
    { // Matrix cannot be p.d.:
      useCholesky= false;
      return;
    }
  
  // If a 9-point stencil is required, and if nw=sw=ne=se does _not_
  // hold, and if there is at least one Neumann boundary, then we
  // might have problems scaling the equations corresponding to
  // Neumann boundary points:
  if (ncoeff==9) {
    if (!(matcoeff[0]==matcoeff[2] &&
          matcoeff[0]==matcoeff[6] &&
	  matcoeff[0]==matcoeff[8]))
      // nw=ne=sw=se does _not_ hold!
      if (btypes[dpNORTH]==NEUMANN || btypes[dpEAST]==NEUMANN ||
          btypes[dpSOUTH]==NEUMANN || btypes[dpWEST]==NEUMANN) {
        // There is at least one Neumann boundary:
        useCholesky= false;
        return;
      }
  }
 
  for(i= 0; (i<ncoeff) && (i!=center); i++)
    ndiagsum+= fabs(matcoeff[i]);

  if (ndiagsum<=matcoeff[center])
    { // 1. We assume the matrix to be irreducible
      // 2. Strict inequalities hold for the grid nodes close to the
      // Dirichlet boundaries (there must be at least one)
      useCholesky= true;
      return;
    }

  // Otherwise, we treat the matrix as if it were not s.p.d.:
  useCholesky= false;
  return;
}


// Function dpDirectSolve
// ----------------------
// Solves the coarsest system of linear equations directly using LAPACK
// routines, note that since boundary conditions on coarser grids are
// always homogeneous, the rhs does not have to be modified after
// restricting the residuals to the coarsest grid
//
// Parameters:
// -----------
// u: Solution vector
// f: Rhs
 
void dpDirectSolve(DIME_REAL *u, DIME_REAL *f, int nxp, int nyp, tBoundary *bTypes)
{
  int i, nrhs= 1, info= 0;
  char c= 'N';
  char uplo= 'U';

  // If a Cholesky decomposition has been performed, we have scale the right-hand
  // sides corresponding to Neumann boundary points, since the corresponding matrix
  // entries have been scaled due to symmetry reasons:
  if (useCholesky==true) {
    // South-west corner:
    if (bTypes[dpSOUTH]==NEUMANN && bTypes[dpWEST]==NEUMANN)
      f[0]*= 0.25;
    // South-east corner:
    if (bTypes[dpSOUTH]==NEUMANN && bTypes[dpEAST]==NEUMANN)
      f[nxp-1]*= 0.25;
    // North-west corner:
    if (bTypes[dpNORTH]==NEUMANN && bTypes[dpWEST]==NEUMANN)
      f[(nyp-1)*nxp]*= 0.25;
    // North-east corner:
    if (bTypes[dpNORTH]==NEUMANN && bTypes[dpEAST]==NEUMANN)
      f[nxp*nyp-1]*= 0.25;
    // Southern boundary line:
    if (bTypes[dpSOUTH]==NEUMANN)
      for (i= 1; i<nxp-1; i++)
	f[i]*= 0.5;
    // Western boundary line:
    if (bTypes[dpWEST]==NEUMANN)
      for (i= 1; i<nyp-1; i++)
	f[i*nxp]*= 0.5;
    // Eastern boundary line:
    if (bTypes[dpEAST]==NEUMANN)
      for (i= 2; i<nyp; i++)
	f[i*nxp-1]*= 0.5;
    // Northern boundary line:
    if (bTypes[dpNORTH]==NEUMANN)
      for (i= 1; i<nxp-1; i++)
	f[(nyp-1)*nxp+i]*= 0.5;
  }

  // Set up rhs:
  for(i= 0; i<neqs; i++)
    w[i]= f[eqmap[i]];

  if (useCholesky==true)
    // Call the appropriate LAPACK routine to solve the system:
    dpF77SUBCALL(XPBTRS,`&uplo, &neqs, &kl, &nrhs, Ac, &ldab, w, &neqs, &info');
  else
    // Call the appropriate LAPACK routine to solve the system:
    dpF77SUBCALL(XGBTRS,`&c, &neqs, &kl, &kl, &nrhs, Ac, &ldab, ipiv, w, &neqs, &info');
  
  if (info<0)
    {
      cerr << "DiMEPACK: Internal error: cannot solve coarsest system directly!" << endl;
      exit(1);
    }

  // Copy results into solution vector:
  for(i= 0; i<neqs; i++)
    u[eqmap[i]]= w[i];
  if (fixCorner==true)
    u[0]= 0.0;
  
  return;
}


// Function dpDirectSolveCleanup
// -----------------------------
// Frees the memory needed by the direct solver on the coarsest grid:

void dpDirectSolveCleanup(void)
{
  delete [] eqmap;
  delete [] Ac;
  delete [] w;
  if (useCholesky==false)
    delete [] ipiv;

  return;
}
