include(fortran-interface.m4)

#include <iostream.h>
#include <stdlib.h>
#include <assert.h>
#include "dpDirectSolve.h"
#include "dpDirectSolveGlobals.h"
#include "dpLAPACK.h"


// Macros to compute the positions within the matrices which are passed
// to the LAPACK routines SGBTRF/DGBTRF (LU factorization of a general band
// matrix) and SPBTRF/DPBTRF (Cholesky factorization of a s.p.d. matrix,
// the upper triangle is stored):
#define dpGetIndexGB(a,b) ((b)*ldab)+2*kl+(a)-(b)
#define dpGetIndexPB(a,b) ((b)*ldab)+kl+(a)-(b)

// Choose the correct LAPACK routines depending on the data type that is used
// for floating-point numbers: double precision or single precision:
#ifdef DIME_DOUBLE
dpCreateDefine(XGBTRF,dgbtrf)
dpCreateDefine(XPBTRF,dpbtrf)
#else
dpCreateDefine(XGBTRF,sgbtrf)
dpCreateDefine(XPBTRF,spbtrf)
#endif


// Function dpBuild9pCMatrix
// -------------------------
// Sets up the matrix of the coarsest system in case of a 9-point stencil
// equations corresponding to Neumann boundary points have to be scaled
// appropriately in order to obtain a symmetric matrix for a problem
// with symmetric coefficients 
//
// Parameters:
// -----------
// nxp, nyp:      Numbers of grid points on the coarsest grid
// matcoeff:      Array coefficients (matrix entries)
// omega:         Relaxation parameter
// btypes:        Array of boundary types
// fixCGSolution: Flag specifying if SW corner value is to be fixed on the coarsest grid

void dpBuild9pCMatrix(int nxp, int nyp, DIME_REAL *matcoeff, DIME_REAL omega,
		      tBoundary *btypes, const bool fixCGSolution)
{
  int x, y, info= 0;
  char uplo= 'U'; // The upper triangle is given if Cholesky's method is applied
  int eqc= 0; // Equation counter
  DIME_REAL no, ne, ea, se, so, sw, we, nw, ce;

  // This flag indicates if we have to fix the solution value in the SW corner
  // on the coarsest grid, this may be necessary for certain singular problems:
  fixCorner= fixCGSolution;
  dpCheckProblem(9, matcoeff, btypes);

  // Determine the number of equations to be created:
  neqs= dpGetNumEqs(nxp, nyp, btypes); // Number of equations

  // Depending on the fixCorner flag, there might be no equation for the
  // south-west corner point:
  if(fixCorner==true)
    neqs--;

  // Array needed to map the equations to the grid points:
  eqmap= (int *) new int[neqs];
  assert(eqmap!=NULL);

  // Compute the entries on the bands of the matrix:
  no= matcoeff[7]*omega/matcoeff[4];
  ne= matcoeff[8]*omega/matcoeff[4];
  ea= matcoeff[5]*omega/matcoeff[4];
  se= matcoeff[2]*omega/matcoeff[4];
  so= matcoeff[1]*omega/matcoeff[4];
  sw= matcoeff[0]*omega/matcoeff[4];
  we= matcoeff[3]*omega/matcoeff[4];
  nw= matcoeff[6]*omega/matcoeff[4];
  ce= omega;

  // Determine if Cholesky's method can be applied to factorize the matrix: 
  dpUseCholesky(9, matcoeff, btypes);
  useCholesky= false; // !!!!!!!!!!!!!!!!!!!!!!!!!!!!

#ifdef DIME_DEBUG_DIRECTSOLVE
  if (useCholesky==true)
    cout << "DiMEPACK: Using Cholesky's method" << endl;
#endif

  // Determine number kl of subdiagonals (= number of superdiagonals):
  kl= nxp+1;
  if (btypes[dpEAST]==DIRICHLET)
    --kl; // East boundary is Dirichlet boundary
  if (btypes[dpWEST]==DIRICHLET)
    --kl; // West boundary is Dirichlet boundary

  if (useCholesky==true)
    { // ********** PREPARE AND PERFORM CHOLESKY FACTORIZATION: **********
      // Allocate and initialize memory for the matrix:
      ldab= kl+1;
      Ac= (DIME_REAL *) new DIME_REAL[ldab*neqs];
      w= (DIME_REAL *) new DIME_REAL[neqs];
      assert(Ac!=NULL);
      assert(w!=NULL);

      for(x= 0; x<ldab*neqs; x++)
	Ac[x]= 0.0;

      // South-west corner (if two Neumann boundaries meet and the user
      // wants to fix the solution, there will be no equation for the
      // south-west corner point (this value will be set to 0, by convention):
      if (fixCorner==false &&
	  btypes[dpSOUTH]==NEUMANN && btypes[dpWEST]==NEUMANN)
	{
	  Ac[dpGetIndexPB(eqc,eqc)]= ce*0.25;
	  Ac[dpGetIndexPB(eqc,eqc+1)]= (ea+we)*0.25;
	  Ac[dpGetIndexPB(eqc,eqc+kl-1)]= (no+so)*0.25;
	  Ac[dpGetIndexPB(eqc,eqc+kl)]= (nw+ne+se+sw)*0.25;
	  eqmap[eqc]= 0;
	  ++eqc;
	}

      // Eastern neighbor of south-west corner:
      if (btypes[dpSOUTH]==NEUMANN)
	{
	  Ac[dpGetIndexPB(eqc,eqc)]= ce*0.5;
	  Ac[dpGetIndexPB(eqc,eqc+kl-1)]= (no+so)*0.5;
	  if (btypes[dpWEST]==NEUMANN) {
            Ac[dpGetIndexPB(eqc,eqc+kl-2)]= (sw+nw)*0.5;
          }  
	  if (nxp>3 || btypes[dpEAST]==NEUMANN) {
	    Ac[dpGetIndexPB(eqc,eqc+1)]= ea*0.5;
	    Ac[dpGetIndexPB(eqc,eqc+kl)]= (se+ne)*0.5;
          }
	  eqmap[eqc]= 1;
	  ++eqc;
	}
	  
      // Interior southern boundary points:
      if (btypes[dpSOUTH]==NEUMANN)
	for(x= 2; x<nxp-2; x++)
	  {
	    Ac[dpGetIndexPB(eqc,eqc)]= ce*0.5;
	    Ac[dpGetIndexPB(eqc,eqc+1)]= ea*0.5;
	    Ac[dpGetIndexPB(eqc,eqc+kl-1)]= (no+so)*0.5;
	    Ac[dpGetIndexPB(eqc,eqc+kl-2)]= (sw+nw)*0.5;
	    Ac[dpGetIndexPB(eqc,eqc+kl)]= (se+ne)*0.5;
	    eqmap[eqc]= x;
	    ++eqc;
	  }

      // Western neighbor of south-east corner:
      if (nxp>3 && btypes[dpSOUTH]==NEUMANN)
	{
	  Ac[dpGetIndexPB(eqc,eqc)]= ce*0.5;
	  Ac[dpGetIndexPB(eqc,eqc+kl-1)]= (no+so)*0.5;
	  Ac[dpGetIndexPB(eqc,eqc+kl-2)]= (sw+nw)*0.5;
	  if (btypes[dpEAST]==NEUMANN) {
	    Ac[dpGetIndexPB(eqc,eqc+1)]= ea*0.5;
	    Ac[dpGetIndexPB(eqc,eqc+kl)]= (se+ne)*0.5;
          }
	  eqmap[eqc]= nxp-2;
	  ++eqc;
	}

      // South-east corner:
      if (btypes[dpEAST]==NEUMANN && btypes[dpSOUTH]==NEUMANN)
	{
	  Ac[dpGetIndexPB(eqc,eqc)]= ce*0.25;
	  Ac[dpGetIndexPB(eqc,eqc+kl-1)]= (no+so)*0.25;
	  Ac[dpGetIndexPB(eqc,eqc+kl-2)]= (nw+ne+se+sw)*0.25;
	  eqmap[eqc]= nxp-1;
	  ++eqc;
	}

      // Northern neighbor of south-west corner:
      if (btypes[dpWEST]==NEUMANN)
	{
	  Ac[dpGetIndexPB(eqc,eqc)]= ce*0.5;
	  Ac[dpGetIndexPB(eqc,eqc+1)]= (we+ea)*0.5;
	  if(nyp>3 || btypes[dpNORTH]==NEUMANN) {
	    Ac[dpGetIndexPB(eqc,eqc+kl-1)]= no*0.5;
	    Ac[dpGetIndexPB(eqc,eqc+kl)]= (ne+nw)*0.5;
          }
	  eqmap[eqc]= nxp;
	  ++eqc;
	}
      
      // Interior south-west corner:
      Ac[dpGetIndexPB(eqc,eqc)]= ce;
      if (nyp>3 || btypes[dpNORTH]==NEUMANN) {
	Ac[dpGetIndexPB(eqc,eqc+kl-1)]= no;
        if (btypes[dpWEST]==NEUMANN)
          Ac[dpGetIndexPB(eqc,eqc+kl-2)]= nw;
      }
      if (nxp>3 || btypes[dpEAST]==NEUMANN) {
	Ac[dpGetIndexPB(eqc,eqc+1)]= ea;
        if (nyp>3 || btypes[dpNORTH]==NEUMANN)
          Ac[dpGetIndexPB(eqc,eqc+kl)]= ne;
      }
      eqmap[eqc]= nxp+1;
      ++eqc;
      
      // Interior points of first interior row:
      for (x= nxp+2; x<2*nxp-2; x++)
	{
	  Ac[dpGetIndexPB(eqc,eqc)]= ce;
	  Ac[dpGetIndexPB(eqc,eqc+1)]= ea;
	  if (nyp>3 || btypes[dpNORTH]==NEUMANN) {
	    Ac[dpGetIndexPB(eqc,eqc+kl-1)]= no;
	    Ac[dpGetIndexPB(eqc,eqc+kl-2)]= nw;
	    Ac[dpGetIndexPB(eqc,eqc+kl)]= ne;
          }
	  eqmap[eqc]= x;
	  ++eqc;
	}

      // Interior south-east corner:
      if (nxp>3)
	{
	  Ac[dpGetIndexPB(eqc,eqc)]= ce;
	  if (btypes[dpEAST]==NEUMANN)
	    Ac[dpGetIndexPB(eqc,eqc+1)]= ea;
	  if (nyp>3 || btypes[dpNORTH]==NEUMANN) {
	    Ac[dpGetIndexPB(eqc,eqc+kl-1)]= no;
            Ac[dpGetIndexPB(eqc,eqc+kl-2)]= nw;
            if (btypes[dpEAST]==NEUMANN)
              Ac[dpGetIndexPB(eqc,eqc+kl)]= ne;
          }
	  eqmap[eqc]= 2*nxp-2;
	  ++eqc;
	}

      // Northern neighbor of south-east corner:
      if (btypes[dpEAST]==NEUMANN)
	{
	  Ac[dpGetIndexPB(eqc,eqc)]= ce*0.5;
	  if (nyp>3 || btypes[dpNORTH]==NEUMANN) {
	    Ac[dpGetIndexPB(eqc,eqc+kl-1)]= no*0.5;
	    Ac[dpGetIndexPB(eqc,eqc+kl-2)]= (ne+nw)*0.5;
          }
	  eqmap[eqc]= 2*nxp-1;
	  ++eqc;
	}

      // Process grid lines 2 to nyp-3:
      for (y= 2; y<nyp-2; y++)
	{
	  // Western boundary point:
	  if (btypes[dpWEST]==NEUMANN)
	    {
	      Ac[dpGetIndexPB(eqc,eqc)]= ce*0.5;
	      Ac[dpGetIndexPB(eqc,eqc+kl-1)]= no*0.5;
	      Ac[dpGetIndexPB(eqc,eqc+1)]= (ea+we)*0.5;
	      Ac[dpGetIndexPB(eqc,eqc+kl)]= (ne+nw)*0.5;
	      eqmap[eqc]= y*nxp;
	      ++eqc;
	    }

	  // Eastern neighbor of the western boundary point:
	  Ac[dpGetIndexPB(eqc,eqc)]= ce;
	  Ac[dpGetIndexPB(eqc,eqc+kl-1)]= no;
	  if (btypes[dpWEST]==NEUMANN) {
	    Ac[dpGetIndexPB(eqc,eqc+kl-2)]= nw;
          }
	  if(nxp>3 || btypes[dpEAST]==NEUMANN) {
	    Ac[dpGetIndexPB(eqc,eqc+1)]= ea;
	    Ac[dpGetIndexPB(eqc,eqc+kl)]= ne;
          }
	  eqmap[eqc]= y*nxp+1;
	  ++eqc;

	  // Interior points of the current row:
	  for (x= 2; x<nxp-2; x++)
	    {
	      Ac[dpGetIndexPB(eqc,eqc)]= ce;
	      Ac[dpGetIndexPB(eqc,eqc+1)]= ea;
	      Ac[dpGetIndexPB(eqc,eqc+kl-1)]= no;
	      Ac[dpGetIndexPB(eqc,eqc+kl)]= ne;
	      Ac[dpGetIndexPB(eqc,eqc+kl-2)]= nw;
	      eqmap[eqc]= y*nxp+x;
	      ++eqc;
	    }

	  // Western neighbor of the eastern boundary point:
	  if (nxp>3)
	    {
	      Ac[dpGetIndexPB(eqc,eqc)]= ce;
	      Ac[dpGetIndexPB(eqc,eqc+kl-1)]= no;
	      Ac[dpGetIndexPB(eqc,eqc+kl-2)]= nw;
	      if (btypes[dpEAST]==NEUMANN) {
		Ac[dpGetIndexPB(eqc,eqc+1)]= ea;
		Ac[dpGetIndexPB(eqc,eqc+kl)]= ne;
              }
	      eqmap[eqc]= (y+1)*nxp-2;
	      ++eqc;
	    }

	  // Eastern boundary point:
	  if (btypes[dpEAST]==NEUMANN)
	    {
	      Ac[dpGetIndexPB(eqc,eqc)]= ce*0.5;
	      Ac[dpGetIndexPB(eqc,eqc+kl-1)]= no*0.5;
	      Ac[dpGetIndexPB(eqc,eqc+kl-2)]= (ne+nw)*0.5;
	      eqmap[eqc]= (y+1)*nxp-1;
	      ++eqc;
	    }
	}

      // Southern neighbor of north-west corner:
      if (nyp>3 && btypes[dpWEST]==NEUMANN)
	{
	  Ac[dpGetIndexPB(eqc,eqc)]= ce*0.5;
	  Ac[dpGetIndexPB(eqc,eqc+1)]= (ea+we)*0.5;
	  if (btypes[dpNORTH]==NEUMANN) {
	    Ac[dpGetIndexPB(eqc,eqc+kl-1)]= no*0.5;
	    Ac[dpGetIndexPB(eqc,eqc+kl)]= (ne+nw)*0.5;
          }
	  eqmap[eqc]= (nyp-2)*nxp;
	  ++eqc;
	}
	      
      // Interior north-west corner:
      if (nyp>3)
	{
	  Ac[dpGetIndexPB(eqc,eqc)]= ce;
	  if (btypes[dpNORTH]==NEUMANN) {
	    Ac[dpGetIndexPB(eqc,eqc+kl-1)]= no;
            if (btypes[dpWEST]==NEUMANN)
              Ac[dpGetIndexPB(eqc,eqc+kl-2)]= nw;
          }
	  if (nxp>3 || btypes[dpEAST]==NEUMANN) {
	    Ac[dpGetIndexPB(eqc,eqc+1)]= ea;
            if (btypes[dpNORTH]==NEUMANN)
	      Ac[dpGetIndexPB(eqc,eqc+kl)]= ne;
          }
	  eqmap[eqc]= (nyp-2)*nxp+1;
	  ++eqc;
	}

      // Interior points of last interior row:
      if (nyp>3)
	{
	  for (x= (nyp-2)*nxp+2; x<(nyp-1)*nxp-2; x++)
	    {
	      Ac[dpGetIndexPB(eqc,eqc)]= ce;
	      Ac[dpGetIndexPB(eqc,eqc+1)]= ea;
	      if (btypes[dpNORTH]==NEUMANN) {
		Ac[dpGetIndexPB(eqc,eqc+kl-1)]= no;
		Ac[dpGetIndexPB(eqc,eqc+kl-2)]= nw;
		Ac[dpGetIndexPB(eqc,eqc+kl)]= ne;
              }
	      eqmap[eqc]= x;
	      ++eqc;
	    }
	}

      // Interior north-east corner:
      if (nxp>3 && nyp>3)
	{
	  Ac[dpGetIndexPB(eqc,eqc)]= ce;
	  if (btypes[dpEAST]==NEUMANN)
	    Ac[dpGetIndexPB(eqc,eqc+1)]= ea;
	  if (btypes[dpNORTH]==NEUMANN) {
	    Ac[dpGetIndexPB(eqc,eqc+kl-1)]= no;
	    Ac[dpGetIndexPB(eqc,eqc+kl-2)]= nw;
            if (btypes[dpEAST]==NEUMANN)
	      Ac[dpGetIndexPB(eqc,eqc+kl)]= ne;
          }
	  eqmap[eqc]= (nyp-1)*nxp-2;
	  ++eqc;	  
	}

      // Southern neighbor of north-east corner:
      if (nyp>3 && btypes[dpEAST]==NEUMANN)
	{
	  Ac[dpGetIndexPB(eqc,eqc)]= ce*0.5;
	  if (btypes[dpNORTH]==NEUMANN) {
	    Ac[dpGetIndexPB(eqc,eqc+kl-1)]= no*0.5;
	    Ac[dpGetIndexPB(eqc,eqc+kl-2)]= (ne+nw)*0.5;
          }
	  eqmap[eqc]= (nyp-1)*nxp-1;
	  ++eqc;
	}

      // North-west corner:
      if (btypes[dpNORTH]==NEUMANN && btypes[dpWEST]==NEUMANN)
	{
	  Ac[dpGetIndexPB(eqc,eqc)]= ce*0.25;
	  Ac[dpGetIndexPB(eqc,eqc+1)]= (ea+we)*0.25;
	  eqmap[eqc]= (nyp-1)*nxp;
	  ++eqc;
	}

      // Eastern neighbor of north-west corner:
      if (btypes[dpNORTH]==NEUMANN)
	{
	  Ac[dpGetIndexPB(eqc,eqc)]= ce*0.5;
	  if (nxp>3 || btypes[dpEAST]==NEUMANN)
	    Ac[dpGetIndexPB(eqc,eqc+1)]= ea*0.5;
	  eqmap[eqc]= (nyp-1)*nxp+1;
	  ++eqc;
	}

      // Interior northern boundary:
      if (btypes[dpNORTH]==NEUMANN)
	for(x= (nyp-1)*nxp+2; x<nyp*nxp-2; x++)
	  {
	    Ac[dpGetIndexPB(eqc,eqc)]= ce*0.5;
	    Ac[dpGetIndexPB(eqc,eqc+1)]= ea*0.5;
	    eqmap[eqc]= x;
	    ++eqc;
	  }

      // Western neighbor of north-east corner:
      if (nxp>3 && btypes[dpNORTH]==NEUMANN)
	{
	  Ac[dpGetIndexPB(eqc,eqc)]= ce*0.5;
	  if (btypes[dpEAST]==NEUMANN)
	    Ac[dpGetIndexPB(eqc,eqc+1)]= ea*0.5;
	  eqmap[eqc]= nyp*nxp-2;
	  ++eqc;
	}

      // North-east corner:
      if (btypes[dpNORTH]==NEUMANN && btypes[dpEAST]==NEUMANN)
	{
	  Ac[dpGetIndexPB(eqc,eqc)]= ce*0.25;
	  eqmap[eqc]= nxp*nyp-1;
	  ++eqc;
	}

      // Call the LAPACK routine DGBTRF to factorize the matrix:
      dpF77SUBCALL(XPBTRF,`&uplo, &neqs, &kl, Ac, &ldab, &info');

      if (info!=0)
	{
	  cerr << "DiMEPACK: Internal error: cannot factorize matrix!" << endl;
	  exit(1);
	}
    }
  else
    { // ********** PREPARE AND PERFORM LU FACTORIZATION: **********
      // Allocate and initialize memory for the matrix:
      ldab= 3*kl+1; // Leading dimension of the matrix array
      Ac= (DIME_REAL *) new DIME_REAL[ldab*neqs];
      w= (DIME_REAL *) new DIME_REAL[neqs];
      assert(Ac!=NULL);
      assert(w!=NULL);

      for(x= 0; x<ldab*neqs; x++)
	Ac[x]= 0.0;

      // Array needed to keep the pivot indices:
      ipiv= (int *) new int[neqs];
      assert(ipiv!=NULL);

      // South-west corner (if two Neumann boundaries meet and the user
      // wants to fix the solution, there will be no equation for the
      // south-west corner point (this value will be set to 0, by convention):
      if (fixCorner==false &&
          btypes[dpSOUTH]==NEUMANN && btypes[dpWEST]==NEUMANN)
	{
	  Ac[dpGetIndexGB(eqc,eqc)]= ce;
	  Ac[dpGetIndexGB(eqc,eqc+1)]= ea+we;
	  Ac[dpGetIndexGB(eqc,eqc+kl-1)]= no+so;
	  Ac[dpGetIndexGB(eqc,eqc+kl)]= nw+ne+se+sw;
	  eqmap[eqc]= 0;
	  ++eqc;
	}

      // Eastern neighbor of south-west corner:
      if (btypes[dpSOUTH]==NEUMANN)
	{
	  Ac[dpGetIndexGB(eqc,eqc)]= ce;
	  Ac[dpGetIndexGB(eqc,eqc+kl-1)]= no+so;
	  if (btypes[dpWEST]==NEUMANN) {
            if (fixCorner==false)
              Ac[dpGetIndexGB(eqc,eqc-1)]= we;
            Ac[dpGetIndexGB(eqc,eqc+kl-2)]= sw+nw;
          }  
	  if (nxp>3 || btypes[dpEAST]==NEUMANN) {
	    Ac[dpGetIndexGB(eqc,eqc+1)]= ea;
	    Ac[dpGetIndexGB(eqc,eqc+kl)]= se+ne;
          }
	  eqmap[eqc]= 1;
	  ++eqc;
	}
	  
      // Interior southern boundary points:
      if (btypes[dpSOUTH]==NEUMANN)
	for(x= 2; x<nxp-2; x++)
	  {
	    Ac[dpGetIndexGB(eqc,eqc)]= ce;
	    Ac[dpGetIndexGB(eqc,eqc-1)]= we;
	    Ac[dpGetIndexGB(eqc,eqc+1)]= ea;
	    Ac[dpGetIndexGB(eqc,eqc+kl-1)]= no+so;
	    Ac[dpGetIndexGB(eqc,eqc+kl-2)]= sw+nw;
	    Ac[dpGetIndexGB(eqc,eqc+kl)]= se+ne;
	    eqmap[eqc]= x;
	    ++eqc;
	  }

      // Western neighbor of south-east corner:
      if (nxp>3 && btypes[dpSOUTH]==NEUMANN)
	{
	  Ac[dpGetIndexGB(eqc,eqc)]= ce;
	  Ac[dpGetIndexGB(eqc,eqc-1)]= we;
	  Ac[dpGetIndexGB(eqc,eqc+kl-1)]= no+so;
	  Ac[dpGetIndexGB(eqc,eqc+kl-2)]= sw+nw;
	  if (btypes[dpEAST]==NEUMANN) {
	    Ac[dpGetIndexGB(eqc,eqc+1)]= ea;
	    Ac[dpGetIndexGB(eqc,eqc+kl)]= se+ne;
          }
	  eqmap[eqc]= nxp-2;
	  ++eqc;
	}

      // South-east corner:
      if (btypes[dpEAST]==NEUMANN && btypes[dpSOUTH]==NEUMANN)
	{
	  Ac[dpGetIndexGB(eqc,eqc)]= ce;
	  Ac[dpGetIndexGB(eqc,eqc-1)]= ea+we;
	  Ac[dpGetIndexGB(eqc,eqc+kl-1)]= no+so;
	  Ac[dpGetIndexGB(eqc,eqc+kl-2)]= nw+ne+se+sw;
	  eqmap[eqc]= nxp-1;
	  ++eqc;
	}

      // Northern neighbor of south-west corner:
      if (btypes[dpWEST]==NEUMANN)
	{
	  Ac[dpGetIndexGB(eqc,eqc)]= ce;
	  Ac[dpGetIndexGB(eqc,eqc+1)]= we+ea;
	  if(btypes[dpSOUTH]==NEUMANN) {
	    if (fixCorner==false)
              Ac[dpGetIndexGB(eqc,eqc-kl+1)]= so;
	    Ac[dpGetIndexGB(eqc,eqc-kl+2)]= se+sw;
          }
	  if(nyp>3 || btypes[dpNORTH]==NEUMANN) {
	    Ac[dpGetIndexGB(eqc,eqc+kl-1)]= no;
	    Ac[dpGetIndexGB(eqc,eqc+kl)]= ne+nw;
          }
	  eqmap[eqc]= nxp;
	  ++eqc;
	}
      
      // Interior south-west corner:
      Ac[dpGetIndexGB(eqc,eqc)]= ce;
      if (btypes[dpWEST]==NEUMANN)
	Ac[dpGetIndexGB(eqc,eqc-1)]= we;
      if (btypes[dpSOUTH]==NEUMANN) {
	Ac[dpGetIndexGB(eqc,eqc-kl+1)]= so;
        if (btypes[dpWEST]==NEUMANN && fixCorner==false)
          Ac[dpGetIndexGB(eqc,eqc-kl)]= sw;
      }
      if (nyp>3 || btypes[dpNORTH]==NEUMANN) {
	Ac[dpGetIndexGB(eqc,eqc+kl-1)]= no;
        if (btypes[dpWEST]==NEUMANN)
          Ac[dpGetIndexGB(eqc,eqc+kl-2)]= nw;
      }
      if (nxp>3 || btypes[dpEAST]==NEUMANN) {
	Ac[dpGetIndexGB(eqc,eqc+1)]= ea;
        if (nyp>3 || btypes[dpNORTH]==NEUMANN)
          Ac[dpGetIndexGB(eqc,eqc+kl)]= ne;
        if (btypes[dpSOUTH]==NEUMANN)
          Ac[dpGetIndexGB(eqc,eqc-kl+2)]= se;         
      }
      eqmap[eqc]= nxp+1;
      ++eqc;
      
      // Interior points of first interior row:
      for (x= nxp+2; x<2*nxp-2; x++)
	{
	  Ac[dpGetIndexGB(eqc,eqc)]= ce;
	  Ac[dpGetIndexGB(eqc,eqc-1)]= we;
	  Ac[dpGetIndexGB(eqc,eqc+1)]= ea;
	  if (btypes[dpSOUTH]==NEUMANN) {
	    Ac[dpGetIndexGB(eqc,eqc-kl+1)]= so;
	    Ac[dpGetIndexGB(eqc,eqc-kl)]= sw;
	    Ac[dpGetIndexGB(eqc,eqc-kl+2)]= se;
          }
	  if (nyp>3 || btypes[dpNORTH]==NEUMANN) {
	    Ac[dpGetIndexGB(eqc,eqc+kl-1)]= no;
	    Ac[dpGetIndexGB(eqc,eqc+kl-2)]= nw;
	    Ac[dpGetIndexGB(eqc,eqc+kl)]= ne;
          }
	  eqmap[eqc]= x;
	  ++eqc;
	}

      // Interior south-east corner:
      if (nxp>3)
	{
	  Ac[dpGetIndexGB(eqc,eqc)]= ce;
	  if (btypes[dpEAST]==NEUMANN)
	    Ac[dpGetIndexGB(eqc,eqc+1)]= ea;
	  Ac[dpGetIndexGB(eqc,eqc-1)]= we;
	  if (btypes[dpSOUTH]==NEUMANN) {
	    Ac[dpGetIndexGB(eqc,eqc-kl+1)]= so;
	    Ac[dpGetIndexGB(eqc,eqc-kl)]= sw;
            if (btypes[dpEAST]==NEUMANN)
              Ac[dpGetIndexGB(eqc,eqc-kl+2)]= se;
          }
	  if (nyp>3 || btypes[dpNORTH]==NEUMANN) {
	    Ac[dpGetIndexGB(eqc,eqc+kl-1)]= no;
            Ac[dpGetIndexGB(eqc,eqc+kl-2)]= nw;
            if (btypes[dpEAST]==NEUMANN)
              Ac[dpGetIndexGB(eqc,eqc+kl)]= ne;
          }
	  eqmap[eqc]= 2*nxp-2;
	  ++eqc;
	}

      // Northern neighbor of south-east corner:
      if (btypes[dpEAST]==NEUMANN)
	{
	  Ac[dpGetIndexGB(eqc,eqc)]= ce;
	  Ac[dpGetIndexGB(eqc,eqc-1)]= ea+we;
	  if (btypes[dpSOUTH]==NEUMANN) {
	    Ac[dpGetIndexGB(eqc,eqc-kl+1)]= so;
	    Ac[dpGetIndexGB(eqc,eqc-kl)]= se+sw;
          }
	  if (nyp>3 || btypes[dpNORTH]==NEUMANN) {
	    Ac[dpGetIndexGB(eqc,eqc+kl-1)]= no;
	    Ac[dpGetIndexGB(eqc,eqc+kl-2)]= ne+nw;
          }
	  eqmap[eqc]= 2*nxp-1;
	  ++eqc;
	}

      // Process grid lines 2 to nyp-3:
      for (y= 2; y<nyp-2; y++)
	{
	  // Western boundary point:
	  if (btypes[dpWEST]==NEUMANN)
	    {
	      Ac[dpGetIndexGB(eqc,eqc)]= ce;
	      Ac[dpGetIndexGB(eqc,eqc-kl+1)]= so;
	      Ac[dpGetIndexGB(eqc,eqc+kl-1)]= no;
	      Ac[dpGetIndexGB(eqc,eqc+1)]= ea+we;
	      Ac[dpGetIndexGB(eqc,eqc+kl)]= ne+nw;
	      Ac[dpGetIndexGB(eqc,eqc-kl+2)]= se+sw;
	      eqmap[eqc]= y*nxp;
	      ++eqc;
	    }

	  // Eastern neighbor of the western boundary point:
	  Ac[dpGetIndexGB(eqc,eqc)]= ce;
	  Ac[dpGetIndexGB(eqc,eqc-kl+1)]= so;
	  Ac[dpGetIndexGB(eqc,eqc+kl-1)]= no;
	  if (btypes[dpWEST]==NEUMANN) {
	    Ac[dpGetIndexGB(eqc,eqc-1)]= we;
	    Ac[dpGetIndexGB(eqc,eqc+kl-2)]= nw;
	    Ac[dpGetIndexGB(eqc,eqc-kl)]= sw;
          }
	  if(nxp>3 || btypes[dpEAST]==NEUMANN) {
	    Ac[dpGetIndexGB(eqc,eqc+1)]= ea;
	    Ac[dpGetIndexGB(eqc,eqc+kl)]= ne;
	    Ac[dpGetIndexGB(eqc,eqc-kl+2)]= se;
          }
	  eqmap[eqc]= y*nxp+1;
	  ++eqc;

	  // Interior points of the current row:
	  for (x= 2; x<nxp-2; x++)
	    {
	      Ac[dpGetIndexGB(eqc,eqc)]= ce;
	      Ac[dpGetIndexGB(eqc,eqc-1)]= we;
	      Ac[dpGetIndexGB(eqc,eqc+1)]= ea;
	      Ac[dpGetIndexGB(eqc,eqc-kl+1)]= so;
	      Ac[dpGetIndexGB(eqc,eqc+kl-1)]= no;
	      Ac[dpGetIndexGB(eqc,eqc+kl)]= ne;
	      Ac[dpGetIndexGB(eqc,eqc-kl+2)]= se;
	      Ac[dpGetIndexGB(eqc,eqc-kl)]= sw;
	      Ac[dpGetIndexGB(eqc,eqc+kl-2)]= nw;
	      eqmap[eqc]= y*nxp+x;
	      ++eqc;
	    }

	  // Western neighbor of the eastern boundary point:
	  if (nxp>3)
	    {
	      Ac[dpGetIndexGB(eqc,eqc)]= ce;
	      Ac[dpGetIndexGB(eqc,eqc+kl-1)]= no;
	      Ac[dpGetIndexGB(eqc,eqc-kl+1)]= so;
	      Ac[dpGetIndexGB(eqc,eqc-1)]= we;
	      Ac[dpGetIndexGB(eqc,eqc+kl-2)]= nw;
	      Ac[dpGetIndexGB(eqc,eqc-kl)]= sw;
	      if (btypes[dpEAST]==NEUMANN) {
		Ac[dpGetIndexGB(eqc,eqc+1)]= ea;
		Ac[dpGetIndexGB(eqc,eqc+kl)]= ne;
		Ac[dpGetIndexGB(eqc,eqc-kl+2)]= se;
              }
	      eqmap[eqc]= (y+1)*nxp-2;
	      ++eqc;
	    }

	  // Eastern boundary point:
	  if (btypes[dpEAST]==NEUMANN)
	    {
	      Ac[dpGetIndexGB(eqc,eqc)]= ce;
	      Ac[dpGetIndexGB(eqc,eqc-1)]= ea+we;
	      Ac[dpGetIndexGB(eqc,eqc+kl-1)]= no;
	      Ac[dpGetIndexGB(eqc,eqc-kl+1)]= so;
	      Ac[dpGetIndexGB(eqc,eqc+kl-2)]= ne+nw;
	      Ac[dpGetIndexGB(eqc,eqc-kl)]= se+sw;
	      eqmap[eqc]= (y+1)*nxp-1;
	      ++eqc;
	    }
	}

      // Southern neighbor of north-west corner:
      if (nyp>3 && btypes[dpWEST]==NEUMANN)
	{
	  Ac[dpGetIndexGB(eqc,eqc)]= ce;
	  Ac[dpGetIndexGB(eqc,eqc-kl+1)]= so;
	  Ac[dpGetIndexGB(eqc,eqc+1)]= ea+we;
	  Ac[dpGetIndexGB(eqc,eqc-kl+2)]= se+sw;
	  if (btypes[dpNORTH]==NEUMANN) {
	    Ac[dpGetIndexGB(eqc,eqc+kl-1)]= no;
	    Ac[dpGetIndexGB(eqc,eqc+kl)]= ne+nw;
          }
	  eqmap[eqc]= (nyp-2)*nxp;
	  ++eqc;
	}
	      
      // Interior north-west corner:
      if (nyp>3)
	{
	  Ac[dpGetIndexGB(eqc,eqc)]= ce;
	  Ac[dpGetIndexGB(eqc,eqc-kl+1)]= so;
	  if (btypes[dpWEST]==NEUMANN) {
	    Ac[dpGetIndexGB(eqc,eqc-1)]= we;
	    Ac[dpGetIndexGB(eqc,eqc-kl)]= sw;
          }
	  if (btypes[dpNORTH]==NEUMANN) {
	    Ac[dpGetIndexGB(eqc,eqc+kl-1)]= no;
            if (btypes[dpWEST]==NEUMANN)
              Ac[dpGetIndexGB(eqc,eqc+kl-2)]= nw;
          }
	  if (nxp>3 || btypes[dpEAST]==NEUMANN) {
	    Ac[dpGetIndexGB(eqc,eqc+1)]= ea;
	    Ac[dpGetIndexGB(eqc,eqc-kl+2)]= se;
            if (btypes[dpNORTH]==NEUMANN)
	      Ac[dpGetIndexGB(eqc,eqc+kl)]= ne;
          }
	  eqmap[eqc]= (nyp-2)*nxp+1;
	  ++eqc;
	}

      // Interior points of last interior row:
      if (nyp>3)
	{
	  for (x= (nyp-2)*nxp+2; x<(nyp-1)*nxp-2; x++)
	    {
	      Ac[dpGetIndexGB(eqc,eqc)]= ce;
	      Ac[dpGetIndexGB(eqc,eqc-1)]= we;
	      Ac[dpGetIndexGB(eqc,eqc+1)]= ea;
	      Ac[dpGetIndexGB(eqc,eqc-kl+1)]= so;
	      Ac[dpGetIndexGB(eqc,eqc-kl)]= sw;
	      Ac[dpGetIndexGB(eqc,eqc-kl+2)]= se;
	      if (btypes[dpNORTH]==NEUMANN) {
		Ac[dpGetIndexGB(eqc,eqc+kl-1)]= no;
		Ac[dpGetIndexGB(eqc,eqc+kl-2)]= nw;
		Ac[dpGetIndexGB(eqc,eqc+kl)]= ne;
              }
	      eqmap[eqc]= x;
	      ++eqc;
	    }
	}

      // Interior north-east corner:
      if (nxp>3 && nyp>3)
	{
	  Ac[dpGetIndexGB(eqc,eqc)]= ce;
	  Ac[dpGetIndexGB(eqc,eqc-kl+1)]= so;
	  Ac[dpGetIndexGB(eqc,eqc-1)]= we;
	  Ac[dpGetIndexGB(eqc,eqc-kl)]= sw;
	  if (btypes[dpEAST]==NEUMANN) {
	    Ac[dpGetIndexGB(eqc,eqc+1)]= ea;
	    Ac[dpGetIndexGB(eqc,eqc-kl+2)]= se;
          }
	  if (btypes[dpNORTH]==NEUMANN) {
	    Ac[dpGetIndexGB(eqc,eqc+kl-1)]= no;
	    Ac[dpGetIndexGB(eqc,eqc+kl-2)]= nw;
            if (btypes[dpEAST]==NEUMANN)
	      Ac[dpGetIndexGB(eqc,eqc+kl)]= ne;
          }
	  eqmap[eqc]= (nyp-1)*nxp-2;
	  ++eqc;	  
	}

      // Southern neighbor of north-east corner:
      if (nyp>3 && btypes[dpEAST]==NEUMANN)
	{
	  Ac[dpGetIndexGB(eqc,eqc)]= ce;
	  Ac[dpGetIndexGB(eqc,eqc-kl+1)]= so;
	  Ac[dpGetIndexGB(eqc,eqc-1)]= ea+we;
	  Ac[dpGetIndexGB(eqc,eqc-kl)]= se+sw;
	  if (btypes[dpNORTH]==NEUMANN) {
	    Ac[dpGetIndexGB(eqc,eqc+kl-1)]= no;
	    Ac[dpGetIndexGB(eqc,eqc+kl-2)]= ne+nw;
          }
	  eqmap[eqc]= (nyp-1)*nxp-1;
	  ++eqc;
	}

      // North-west corner:
      if (btypes[dpNORTH]==NEUMANN && btypes[dpWEST]==NEUMANN)
	{
	  Ac[dpGetIndexGB(eqc,eqc)]= ce;
	  Ac[dpGetIndexGB(eqc,eqc-kl+1)]= no+so;
	  Ac[dpGetIndexGB(eqc,eqc+1)]= ea+we;
	  Ac[dpGetIndexGB(eqc,eqc-kl+2)]= nw+ne+se+sw;
	  eqmap[eqc]= (nyp-1)*nxp;
	  ++eqc;
	}

      // Eastern neighbor of north-west corner:
      if (btypes[dpNORTH]==NEUMANN)
	{
	  Ac[dpGetIndexGB(eqc,eqc)]= ce;
	  Ac[dpGetIndexGB(eqc,eqc-kl+1)]= no+so;
	  if (btypes[dpWEST]==NEUMANN) {
	    Ac[dpGetIndexGB(eqc,eqc-1)]= we;
	    Ac[dpGetIndexGB(eqc,eqc-kl)]= sw+nw;
          }
	  if (nxp>3 || btypes[dpEAST]==NEUMANN) {
	    Ac[dpGetIndexGB(eqc,eqc+1)]= ea;
	    Ac[dpGetIndexGB(eqc,eqc-kl+2)]= se+ne;
          }
	  eqmap[eqc]= (nyp-1)*nxp+1;
	  ++eqc;
	}

      // Interior northern boundary:
      if (btypes[dpNORTH]==NEUMANN)
	for(x= (nyp-1)*nxp+2; x<nyp*nxp-2; x++)
	  {
	    Ac[dpGetIndexGB(eqc,eqc)]= ce;
	    Ac[dpGetIndexGB(eqc,eqc-1)]= we;
	    Ac[dpGetIndexGB(eqc,eqc+1)]= ea;
	    Ac[dpGetIndexGB(eqc,eqc-kl+1)]= no+so;
	    Ac[dpGetIndexGB(eqc,eqc-kl)]= sw+nw;
	    Ac[dpGetIndexGB(eqc,eqc-kl+2)]= se+ne;
	    eqmap[eqc]= x;
	    ++eqc;
	  }

      // Western neighbor of north-east corner:
      if (nxp>3 && btypes[dpNORTH]==NEUMANN)
	{
	  Ac[dpGetIndexGB(eqc,eqc)]= ce;
	  Ac[dpGetIndexGB(eqc,eqc-1)]= we;
	  Ac[dpGetIndexGB(eqc,eqc-kl+1)]= no+so;
	  Ac[dpGetIndexGB(eqc,eqc-kl)]= sw+nw;
	  if (btypes[dpEAST]==NEUMANN) {
	    Ac[dpGetIndexGB(eqc,eqc+1)]= ea;
	    Ac[dpGetIndexGB(eqc,eqc-kl+2)]= se+ne;
          }
	  eqmap[eqc]= nyp*nxp-2;
	  ++eqc;
	}

      // North-east corner:
      if (btypes[dpNORTH]==NEUMANN && btypes[dpEAST]==NEUMANN)
	{
	  Ac[dpGetIndexGB(eqc,eqc)]= ce;
	  Ac[dpGetIndexGB(eqc,eqc-kl+1)]= no+so;
	  Ac[dpGetIndexGB(eqc,eqc-1)]= ea+we;
	  Ac[dpGetIndexGB(eqc,eqc-kl)]= nw+ne+se+sw;
	  eqmap[eqc]= nxp*nyp-1;
	  ++eqc;
	}

      // Call the LAPACK routine DGBTRF to factorize the matrix:
      dpF77SUBCALL(XGBTRF,`&neqs, &neqs, &kl, &kl, Ac, &ldab, ipiv, &info');

      if (info<0)
	{
	  cerr << "DiMEPACK: Internal error: cannot factorize matrix!" << endl;
	  exit(1);
	}
      
      if (info>0)
	{
	  cerr << "DiMEPACK: Upper triangular matrix is singular!" << endl;
	  cerr << "Forward-backward substitution impossible!" << endl;
	  exit(1);
	}
    }

#ifdef DIME_DEBUG_DIRECTSOLVE
  cout << "DiMEPACK: Created and factorized ";  
  cout << "coarsest system matrix of order " << neqs;
  if (useCholesky==true)
    cout << " (Cholesky)" << endl;
  else
    cout << " (LU)" << endl;
#endif
  
  return;
}
