#include "dpApplyPadding.h"

#include <paddingTools.h>
#include <assert.h>
#include <vector>

#include "dpTypes.h"
#include <cacheInfo.h>

static void dpInitArrayReferences(arrayInfo& aInfo,arrayInfo& bInfo);


int dpCalcPaddingSize(int xdim, int ydim)
{
#ifdef DIME_DISABLE_PADDING
  return 0;
#elif DIME_STATIC_PADDING
  static int last=15;
  last++;
  return last;
#else
  const cacheInfo cInfo=dpGetCacheInfo();

  std::vector<int> lowerBounds(2,0);
  std::vector<int> sizes(2);
  
  assert(xdim>0);
  assert(ydim>0);
  sizes[0]=xdim;
  sizes[1]=ydim;

  std::vector<intraPaddingCondition> padConditions(3);
  padConditions[0]=conditionLinpad1;
  padConditions[1]=conditionLinpad2;
  padConditions[2]=conditionIntrapadLite;
  
  arrayInfo aInfo(sizeof(DIME_REAL),0,sizes,lowerBounds);
  arrayInfo newInfo=getNeededIntraPadding(cInfo,aInfo,padConditions);
    
  int padSize=newInfo.getColSize(1)-aInfo.getColSize(1);
  assert(padSize>=0);

#ifdef DIME_DEBUG_PADDING
  cout << "DiMEPACK DEBUG: padding of size " << padSize;
  cout << " found for grid of size " << xdim << "x" << ydim << endl;
#endif

  return padSize;
#endif
}

void dpApplyPadding(int maxlevel,const int *xdims,const int *ydims,int *upads, int *fpads)
{
#ifdef DIME_DISABLE_PADDING
  for(int i=1; i<maxlevel; i++)
  {
    upads[i]=fpads[i]=0;
  }
#elif defined (DIME_STATIC_PADDING)
  for(int i=1; i<maxlevel; i++)
  {
    upads[i]=16;
    fpads[i]=17;

#ifdef DIME_DEBUG_PADDING
    cout << "DiMEPACK DEBUG: u padding on level " << i << ": " << upads[i] << endl;
    cout << "DiMEPACK DEBUG: f padding on level " << i << ": " << fpads[i] << endl;
#endif
  }
#else
  const cacheInfo cInfo=dpGetCacheInfo();

  // CW: the lower bound of the arrays which store the grid data
  //  are always zero in our case
  std::vector<int> lowerBounds(2,0);

  std::vector<int> sizes(2);

  // CW: attention to array references will be implemented later
  // For now, we just use simple heuristics
  std::vector<intraPaddingCondition> padConditions(4);
  padConditions[0]=conditionLinpad1;
  padConditions[1]=conditionLinpad2;
  padConditions[2]=conditionIntrapadLite;
  padConditions[3]=conditionIntrapad;

  for(int i=1; i<maxlevel; i++)
  {
    assert(xdims[i]>0);
    assert(ydims[i]>0);
    sizes[0]=xdims[i];
    sizes[1]=ydims[i];

    arrayInfo aInfo(sizeof(DIME_REAL),0,sizes,lowerBounds);
    arrayInfo fInfo(sizeof(DIME_REAL),0,sizes,lowerBounds);
    
    dpInitArrayReferences(aInfo,fInfo);
  
    arrayInfo newAInfo=getNeededIntraPadding(cInfo,aInfo,padConditions);
    arrayInfo newFInfo=getNeededIntraPadding(cInfo,fInfo,padConditions);

    upads[i]=newAInfo.getColSize(1)-xdims[i];
    fpads[i]=newFInfo.getColSize(1)-xdims[i];

    if(fpads[i]==upads[i]){
      // if both padding are equal we have to look for a new one
      // since this may caus cross interferance
      fInfo.setColSize(1,newFInfo.getColSize(1)+1); //  try to find a bigger padding which is o.k.
      arrayInfo secondTryInfo=getNeededIntraPadding(cInfo,fInfo,padConditions);
      fpads[i]=secondTryInfo.getColSize(1)-xdims[i];
    }

#ifdef DIME_DEBUG_PADDING
    cout << "DiMEPACK DEBUG: u padding on level " << i << ": " << upads[i] << endl;
    cout << "DiMEPACK DEBUG: f padding on level " << i << ": " << fpads[i] << endl;
#endif

    assert(upads[i]>=0);
    assert(fpads[i]>=0);
  }
#endif  
}


// Padding heuristic tries to tune for 2d sqaure blocking.
// Hopefully, the padding is also good for the other algorithms
static void dpInitArrayReferences(arrayInfo& aInfo,arrayInfo& bInfo)
{
  assert(aInfo.getNumberOfDimensions()==2);
  assert(bInfo.getNumberOfDimensions()==2);

  std::vector<arrayReference> aRefs;
  std::vector<arrayReference> bRefs;

  arrayReference center(2);
  arrayReference north(2);
  arrayReference south(2);
  arrayReference east(2);
  arrayReference west(2);
	
  // CW: initialize all reference constants
  for(int i=0;i<8;i++)
    {
      for(int j=0;j<8;j++)
	{
	  center[0]=-i-j; center[1]=j;  aRefs.push_back(center); bRefs.push_back(center);
	  north[0]=center(0)+1; north[1]=center(1); aRefs.push_back(north);
	  south[0]=center(0)-1; south[1]=center(1); aRefs.push_back(south);
	  east[0]=center(0); east[1]=center(1)+1; aRefs.push_back(east);
	  west[0]=center(0); west[1]=center(1)-1; aRefs.push_back(west);
	}
    }


  // add center values
  aInfo.addReferences(aRefs);
  bInfo.addReferences(bRefs);
      
}

cacheInfo dpGetCacheInfo()
{
  static bool done=false;
  static int cacheSize;
  static int cacheLineSize;

  if(!done){

    const char *cacheSizeStr=getenv("DIME_CACHE_SIZE");
    const char *cacheLineSizeStr=getenv("DIME_CACHE_LINE_SIZE");

    // setting cacheSize value
    if(cacheSizeStr==NULL){
      cout << "DiMEPACK: Environment variable DIME_CACHE_SIZE not set" << endl;
      cout << "DiMEPACK: Using default cache size (8 KB)" << endl;
      cacheSize=8192;
    }
    else{
      cacheSize=atoi(cacheSizeStr);
      if(cacheSize<=0){
	cout << "DiMEPACK: Invalid cache size spezified in DIME_CACHE_SIZE" << endl;
	cout << "DiMEPACK: Using default cache size (8 KB)" << endl;
	cacheSize=8192;
      }
    }

    // setting cacheLineSize value
    if(cacheLineSizeStr==NULL){
      cout << "DiMEPACK: Environment variable DIME_CACHE_LINE_SIZE not set" << endl;
      cout << "DiMEPACK: Using default cache line size (32 byte)" << endl;
      cacheLineSize=32;
    }
    else{
      cacheLineSize=atoi(cacheLineSizeStr);
      if(cacheLineSize<=0 || cacheLineSize>cacheSize){
	cout << "DiMEPACK: Invalid cache line size spezified in DIME_CACHE_LINE_SIZE" << endl;
	cout << "DiMEPACK: Using default cache line size (32 byte)" << endl;
	cacheLineSize=32;
      }
    }

#ifdef DIME_DEBUG_CACHE_PROPERTIES
    cout << "DiMEPACK DEBUG: Cache size = " << cacheSize << " (byte)" << endl;
    cout << "DiMEPACK DEBUG: Cache line size = " << cacheLineSize << " (byte)" << endl;
#endif

    done=true;

  }

  return cacheInfo(cacheSize,cacheLineSize);
}
