// Benchmarks for DiMEPACK library:

#include <iostream.h>
#include <fstream.h>
#include <math.h>
#include <assert.h>
#include <stdlib.h>
#include "dpUtilities.h"
#include "dimepack.h"

// Poisson problems:
void problem1(void); // FMG, 5-point stencil
void problem2(void); // FMG, 9-point stencil
void problem3(void); // Standard MG, 5-point stencil
void problem4(void); // Standard MG, 9-point stencil
// Convection-diffusion equation:
void problem5(void); // Standard MG, 5-point stencil

// Global constants:
const tNorm nType= L2;
const DIME_REAL epsilon= 1e-18;

// Global variables (to be defined by the program parameters):
int nLevels;
int maxIt;
int nxp, nyp;
int nu1, nu2;
int cacheSize, cacheLineLength;
DIME_REAL omega;
tRestrict rType;

int main(int argc, char **argv)
{
  if (argc!=9 && argc!=10){
    cerr << "benchdp: parameter error!" << endl;
    cerr << "Usage: benchdp <nLevels> <maxIt> <nxp> <nyp> <nu1> <nu2> ";
    cerr << "<restrType> <probId> [<omega>]" << endl; 
    exit (1);
  }
  
  nLevels= atoi(argv[1]);
  maxIt= atoi(argv[2]);	
  nxp= atoi(argv[3]);
  nyp= atoi(argv[4]);	
  nu1= atoi(argv[5]);
  nu2= atoi(argv[6]);
  
  if(strcmp(argv[7],"HW")==0)
    rType= HW;
  else {
    if(strcmp(argv[7],"FW")==0)
      rType= FW;
    else {
      cerr << "benchdp: wrong restriction type!" << endl;
      exit(1);
    }
  }
      
  if(argc==10) omega=(DIME_REAL)atof(argv[9]);
  else omega=1.0;


  dpPrintTimes("start");
  switch(atoi(argv[8])) {
  case 1:
    problem1(); break;
  case 2:
    problem2(); break;
  case 3:
    problem3(); break;
  case 4:
    problem4(); break;
  case 5:
    problem5(); break;
  default:
    cerr << "benchdp: no such problem!" << endl;
    break;
  }
  dpPrintTimes("end");
  
  return 0;
}


// *************************************
// * PROBLEM NO. 1: POISSON'S EQUATION *
// *************************************
void problem1(void)
{
  const DIME_REAL hx= 1.0/(DIME_REAL) (nxp-1);
  const DIME_REAL hy= 1.0/(DIME_REAL) (nyp-1);
  const DIME_REAL hsq = (1.0/(DIME_REAL) (nxp-1))*(1.0/(DIME_REAL) (nxp-1));
  int i;
 
  // Solution vector:
  dpGrid2d u(nxp, nyp, hx, hy);
  u.initzero();
 
  // Right-hand side:
  dpGrid2d f(nxp, nyp, hx, hy);
  int x, y;
  for (y= 0; y<nyp; y++)
    for (x= 0; x<nxp; x++)
      f.setval(x,y)= sin(2.0*M_PI*x*hx)*sin(2.0*M_PI*y*hy);
  
  // Matrix coefficients:
  const int ncoeff=5;
  DIME_REAL *matcoeff=new DIME_REAL[ncoeff];
  assert (matcoeff!=NULL);
  matcoeff[0]= -1.0/hsq;
  matcoeff[1]= -1.0/hsq;
  matcoeff[2]= 4.0/hsq;
  matcoeff[3]= -1.0/hsq;
  matcoeff[4]= -1.0/hsq;
  
  // Boundary types:
  tBoundary btypes[4];
  for(i=0; i<4; i++) {
    btypes[i] = DIRICHLET;
  }
  const bool fixSol=false;
  
  // Boundary values:
  DIME_REAL **bvals=new DIME_REAL*[4];
  bvals[dpNORTH]= new DIME_REAL[nxp];
  bvals[dpSOUTH]= new DIME_REAL[nxp];
  bvals[dpEAST]=  new DIME_REAL[nyp];
  bvals[dpWEST]=  new DIME_REAL[nyp];
  for(i=0; i<nxp; i++)
    {
      bvals[dpNORTH][i]= 0.0;
      bvals[dpSOUTH][i]= 0.0;
    }
  for(i=0; i<nyp; i++)
    {
      bvals[dpEAST][i]= 0.0;
      bvals[dpWEST][i]= 0.0;
    }
  

  // Call the DiMEPACK library function:
  dpFMGVcycleConst(nLevels,nType,epsilon,0,(&u),(&f),nu1,nu2,1,ncoeff,
  		   matcoeff,btypes,bvals,rType,omega,fixSol);

  delete[] bvals[dpNORTH]; 
  delete[] bvals[dpEAST]; 
  delete[] bvals[dpSOUTH]; 
  delete[] bvals[dpWEST]; 
  delete[] bvals;
  delete[] matcoeff;
}

// *************************************
// * PROBLEM NO. 2: POISSON'S EQUATION *
// *************************************
void problem2(void)
{
  const DIME_REAL hx= 1.0/(DIME_REAL) (nxp-1);
  const DIME_REAL hy= 1.0/(DIME_REAL) (nyp-1);
  const DIME_REAL hsq = (1.0/(DIME_REAL) (nxp-1))*(1.0/(DIME_REAL) (nxp-1));
  int i;
 
  // Solution vector:
  dpGrid2d u(nxp, nyp, hx, hy);
  u.initzero();
 
  // Right-hand side:
  dpGrid2d f(nxp, nyp, hx, hy);
  int x, y;
  for (y= 0; y<nyp; y++)
    for (x= 0; x<nxp; x++)
      f.setval(x,y)= sin(2.0*M_PI*x*hx)*sin(2.0*M_PI*y*hy);
  
  // Matrix coefficients:
  const int ncoeff=9;
  DIME_REAL *matcoeff=new DIME_REAL[ncoeff];
  assert (matcoeff!=NULL);
  matcoeff[0]= -1.0/(6.0*hsq);
  matcoeff[1]= -4.0/(6.0*hsq);
  matcoeff[2]= -1.0/(6.0*hsq);
  matcoeff[3]= -4.0/(6.0*hsq);
  matcoeff[4]= 20.0/(6.0*hsq);
  matcoeff[5]= -4.0/(6.0*hsq);
  matcoeff[6]= -1.0/(6.0*hsq);
  matcoeff[7]= -4.0/(6.0*hsq);
  matcoeff[8]= -1.0/(6.0*hsq);
  
  // Boundary types:
  tBoundary btypes[4];
  for(i=0; i<4; i++) {
    btypes[i] = DIRICHLET;
  }
  const bool fixSol=false;
  
  // Boundary values:
  DIME_REAL **bvals=new DIME_REAL*[4];
  bvals[dpNORTH]= new DIME_REAL[nxp];
  bvals[dpSOUTH]= new DIME_REAL[nxp];
  bvals[dpEAST]=  new DIME_REAL[nyp];
  bvals[dpWEST]=  new DIME_REAL[nyp];
  for(i=0; i<nxp; i++)
    {
      bvals[dpNORTH][i]= 0.0;
      bvals[dpSOUTH][i]= 0.0;
    }
  for(i=0; i<nyp; i++)
    {
      bvals[dpEAST][i]= 0.0;
      bvals[dpWEST][i]= 0.0;
    }
  

  // Call the DiMEPACK library function:
  dpFMGVcycleConst(nLevels,nType,epsilon,0,(&u),(&f),nu1,nu2,1,ncoeff,
  		   matcoeff,btypes,bvals,rType,omega,fixSol);

  delete[] bvals[dpNORTH]; 
  delete[] bvals[dpEAST]; 
  delete[] bvals[dpSOUTH]; 
  delete[] bvals[dpWEST]; 
  delete[] bvals;
  delete[] matcoeff;
}

// *************************************
// * PROBLEM NO. 3: POISSON'S EQUATION *
// *************************************
void problem3()
{
  const DIME_REAL hx= 1.0/(DIME_REAL) (nxp-1);
  const DIME_REAL hy= hx;
  const DIME_REAL hsq= hx*hx;
  int i;

  // Solution vector:  
  dpGrid2d u(nxp, nyp, hx, hy);
  u.initzero();
  const bool isInitialized=true;
  
  // Right-hand side:
  dpGrid2d f(nxp, nyp, hx, hy);
  int x, y;
  for (y= 0; y<nyp; y++)
    for (x= 0; x<nxp; x++)
      f.setval(x,y)= sin(5*M_PI*x*hx)*sin(M_PI*y*hy);
  
  // Matrix coefficients:
  const int ncoeff=5;
  DIME_REAL *matcoeff=new DIME_REAL[ncoeff];
  assert (matcoeff!=NULL);
  matcoeff[0]= -1.0/hsq;
  matcoeff[1]= -1.0/hsq;
  matcoeff[2]= 4.0/hsq;
  matcoeff[3]= -1.0/hsq;
  matcoeff[4]= -1.0/hsq;
  
  // Boundary types:
  tBoundary btypes[4];
  for(i=0; i<4; i++) {
    btypes[i]= DIRICHLET;
  }
  const bool fixSol=false;    
    
  // Boundary values
  DIME_REAL **bvals=new DIME_REAL*[4];
  bvals[dpNORTH]= new DIME_REAL[nxp];
  bvals[dpSOUTH]= new DIME_REAL[nxp];
  bvals[dpEAST]=  new DIME_REAL[nyp];
  bvals[dpWEST]=  new DIME_REAL[nyp];
  for(i=0; i<nxp; i++) {
    bvals[dpNORTH][i]= 0.0;
    bvals[dpSOUTH][i]= 0.0;
  }
  for(i=0; i<nyp; i++) {
    bvals[dpEAST][i]= 0.0;
    bvals[dpWEST][i]= 0.0;
  }
    
  // Call the DiMEPACK library function:
  dpVcycleConst(nLevels,nType,epsilon,maxIt,(&u),isInitialized,(&f),nu1,nu2,ncoeff,
		matcoeff,btypes,bvals,rType,omega,fixSol);

  delete[] bvals[dpNORTH]; 
  delete[] bvals[dpEAST]; 
  delete[] bvals[dpSOUTH]; 
  delete[] bvals[dpWEST]; 
  delete[] bvals;
  delete[] matcoeff;
}

// *************************************
// * PROBLEM NO. 4: POISSON'S EQUATION *
// *************************************
void problem4()
{
  const DIME_REAL hx= 1.0/(DIME_REAL) (nxp-1);
  const DIME_REAL hy= hx;
  const DIME_REAL hsq= hx*hx;
  int i;

  // Solution vector:  
  dpGrid2d u(nxp, nyp, hx, hy);
  u.initzero();
  const bool isInitialized=true;
  
  // Right-hand side:
  dpGrid2d f(nxp, nyp, hx, hy);
  int x, y;
  for (y= 0; y<nyp; y++)
    for (x= 0; x<nxp; x++)
      f.setval(x,y)= sin(5*M_PI*x*hx)*sin(M_PI*y*hy);

  // Output the rhs for debugging purposes:
  // {
  //  ofstream outf("f-init.dat");
  //  dpPrintGrid(outf, f.getmem(), f.getdimx(), f.getdimy(), f.getpad()); 
  // }
  
  // Matrix coefficients:
  const int ncoeff=9;
  DIME_REAL *matcoeff=new DIME_REAL[ncoeff];
  assert (matcoeff!=NULL);
  matcoeff[0]= -1.0/(6.0*hsq);
  matcoeff[1]= -4.0/(6.0*hsq);
  matcoeff[2]= -1.0/(6.0*hsq);
  matcoeff[3]= -4.0/(6.0*hsq);
  matcoeff[4]= 20.0/(6.0*hsq);
  matcoeff[5]= -4.0/(6.0*hsq);
  matcoeff[6]= -1.0/(6.0*hsq);
  matcoeff[7]= -4.0/(6.0*hsq);
  matcoeff[8]= -1.0/(6.0*hsq);
  
  // Boundary types:
  tBoundary btypes[4];
  for(i=0; i<4; i++) {
    btypes[i]= DIRICHLET;
  }
  const bool fixSol=false;    
    
  // Boundary values
  DIME_REAL **bvals=new DIME_REAL*[4];
  bvals[dpNORTH]= new DIME_REAL[nxp];
  bvals[dpSOUTH]= new DIME_REAL[nxp];
  bvals[dpEAST]=  new DIME_REAL[nyp];
  bvals[dpWEST]=  new DIME_REAL[nyp];
  for(i=0; i<nxp; i++) {
    bvals[dpNORTH][i]= 0.0;
    bvals[dpSOUTH][i]= 0.0;
  }
  for(i=0; i<nyp; i++) {
    bvals[dpEAST][i]= 0.0;
    bvals[dpWEST][i]= 0.0;
  }
    
  // Call the DiMEPACK library function:
  dpVcycleConst(nLevels,nType,epsilon,maxIt,(&u),isInitialized,(&f),nu1,nu2,ncoeff,
		matcoeff,btypes,bvals,rType,omega,fixSol);

  delete[] bvals[dpNORTH]; 
  delete[] bvals[dpEAST]; 
  delete[] bvals[dpSOUTH]; 
  delete[] bvals[dpWEST]; 
  delete[] bvals;
  delete[] matcoeff;
}

// ***********************************************************
// * PROBLEM NO. 5: POISSON'S EQUATION WITH SMALL CONVECTION *
// ***********************************************************
void problem5()
{
  const DIME_REAL hx= 1.0/(DIME_REAL) (nxp-1);
  const DIME_REAL hy= hx;
  const DIME_REAL hsq= hx*hx;
  int i;
  const DIME_REAL conv= 1.0;

  // Solution vector:  
  dpGrid2d u(nxp, nyp, hx, hy);
  u.initzero();
  const bool isInitialized=true;
  
  // Right-hand side:
  dpGrid2d f(nxp, nyp, hx, hy);
  int x, y;
  for (y= 0; y<nyp; y++)
    for (x= 0; x<nxp; x++)
      f.setval(x,y)= 0.0;
  
  // Matrix coefficients:
  const int ncoeff=5;
  DIME_REAL *matcoeff=new DIME_REAL[ncoeff];
  assert (matcoeff!=NULL);

  matcoeff[0]= -1.0/hsq;
  matcoeff[1]= -1.0/hsq;
  // First-order upwind scheme for convective contribution:
  matcoeff[2]= 4.0/hsq + conv/hx;
  matcoeff[3]= -1.0/hsq - conv/hx;
  matcoeff[4]= -1.0/hsq;

  // Boundary types:
  tBoundary btypes[4];
  btypes[dpWEST]= DIRICHLET;
  btypes[dpNORTH]= DIRICHLET;
  btypes[dpSOUTH]= DIRICHLET;
  btypes[dpEAST]= NEUMANN;

  const bool fixSol= false;
    
  // Boundary values:
  DIME_REAL **bvals=new DIME_REAL*[4];
  bvals[dpNORTH]= new DIME_REAL[nxp];
  bvals[dpSOUTH]= new DIME_REAL[nxp];
  bvals[dpEAST]=  new DIME_REAL[nyp];
  bvals[dpWEST]=  new DIME_REAL[nyp];
  for(i=0; i<nxp; i++) {
    bvals[dpNORTH][i]= 0.0;
    bvals[dpSOUTH][i]= 0.0;
  }
  for(i=0; i<nyp; i++) {
    bvals[dpEAST][i]= 0.0;
    bvals[dpWEST][i]= sin(M_PI*i*hy);
  }
    
  // Call the DiMEPACK library function:
  dpVcycleConst(nLevels,nType,epsilon,maxIt,(&u),isInitialized,(&f),nu1,nu2,ncoeff,
		matcoeff,btypes,bvals,rType,omega,fixSol);

  delete[] bvals[dpNORTH]; 
  delete[] bvals[dpEAST]; 
  delete[] bvals[dpSOUTH]; 
  delete[] bvals[dpWEST]; 
  delete[] bvals;
  delete[] matcoeff;
}

// End of file
